// Crimson Fields -- a game of tactical warfare
// Copyright (C) 2000-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// initwindow.cpp
////////////////////////////////////////////////////////////////////////

#include <string.h>

#include "initwindow.h"
#include "filewindow.h"
#include "aiselector.h"
#include "misc.h"
#include "game.h"
#include "options.h"
#include "msgs.h"

#define IW_WIDGET_TYPE		0
#define IW_WIDGET_LOAD		1
#define IW_WIDGET_NEW		2
#define IW_WIDGET_LEVELS	3
#define IW_WIDGET_LANGUAGE	4
#define IW_WIDGET_VIDEO		5
#define IW_WIDGET_START		6

extern Options CFOptions;

struct initLevInfo {    // used as user_data for the LevListWidget
  bool password;
  string *full_path;
};

const short GenericOptionsWindow::B_ID_OK      = 1;

////////////////////////////////////////////////////////////////////////
// NAME       : LevListWidget::PrintItem
// DESCRIPTION: Print a level name. If the map is protected by a
//              password print it in red colour, otherwise in green.
//              Put the default number of players behind the title in
//              brackets.
// PARAMETERS : item - node to print
//              dest - destination surface
//              x    - left edge of printing area
//              y    - top edge of printing area
//              clip - clipping rectangle
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void LevListWidget::PrintItem( const TLWNode *item, Surface *dest,
                               short x, short y, const Rect &clip ) const {
  Color col( CF_COLOR_WHITE );

  if ( item->UserData() ) {
    initLevInfo *info = (initLevInfo *)item->UserData();
    if ( info->password ) col = surface->GetFGPen();
  }

  Rect oldclip;
  dest->GetClipRect( oldclip );
  dest->SetClipRect( clip );
  font->Write( item->Name(), dest, x, y, col );
  dest->SetClipRect( oldclip );
}


////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::InitWindow
// DESCRIPTION: Create the main menu window. The player(s) can select
//              the level they want to play and set other options.
// PARAMETERS : view  - view to attach the window to
//              title - pointer to title window; must be closed when this
//                      window is closed
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

InitWindow::InitWindow( View *view, Window *title ) : Window( WIN_CENTER, view ) {
  this->title = title;

  // read list of maps/saves
  const string home_lev( get_home_levels_dir() );
  if ( !home_lev.empty() )
    FileWindow::CreateFilesList( home_lev.c_str(), ".lev", levels );
  FileWindow::CreateFilesList( get_levels_dir().c_str(), ".lev", levels );
  CompleteFilesList( levels );
  FileWindow::CreateFilesList( get_save_dir().c_str(), ".sav", saves );
  CompleteFilesList( saves );

  Audio::PlayMusic( CF_MUSIC_THEME );

  view->SetFGPen( Color( 0x00d87c00 ) );
  view->SetBGPen( Color( CF_COLOR_BLACK ) );

  // calculate window dimensions
  const char *typelbl = MSG(MSG_B_GAME_TYPE);
  const char *optlbl = MSG(MSG_OPTIONS);
  unsigned short typelen = sfont->TextWidth(typelbl) - sfont->CharWidth('_');
  unsigned short optlen = sfont->TextWidth(optlbl);

  unsigned short xoff = 10 + MAX(typelen, optlen);
  unsigned short wdh = sfont->Height() + 8;

  SetSize( MIN(view->Width(), xoff + sfont->Width() * 40 + 20),
           MIN(view->Height(), wdh * 12 + 10 ) );

  // create widgets
  Widget *wd;

  const char *typelabels[4];
  typelabels[0] = MSG(MSG_GAME_HOT_SEAT);
  typelabels[1] = MSG(MSG_GAME_AI);
  typelabels[2] = MSG(MSG_GAME_PBEM);
  typelabels[3] = MSG(MSG_GAME_AI_VS_AI);
  typelabels[4] = 0;

  gtypewidget = new CycleWidget( IW_WIDGET_TYPE, xoff, 5,
                (w - xoff - 10) / 2, wdh, WIDGET_ALIGN_LEFT,
                typelbl, 0, typelabels, this );
  gtypewidget->SetHook( this );

  newloadwidget = new ButtonWidget( IW_WIDGET_LOAD,
                  gtypewidget->LeftEdge() + gtypewidget->Width() + 5,
                  gtypewidget->TopEdge(), gtypewidget->Width(), wdh,
                  0, MSG(MSG_B_LOAD_GAME), this );
  newloadwidget->SetHook( this );

  wd = new ButtonWidget( IW_WIDGET_LANGUAGE,
                xoff, gtypewidget->TopEdge() + wdh + 5,
                gtypewidget->Width(), wdh, 0, MSG(MSG_B_OPT_LANGUAGE), this );
  wd->SetHook( this );

  wd = new ButtonWidget( IW_WIDGET_VIDEO,
                wd->LeftEdge() + wd->Width() + 5, wd->TopEdge(),
                wd->Width(), wdh, 0, MSG(MSG_B_OPT_VIDEO), this );
  wd->SetHook( this );

  levwidget = new LevListWidget( IW_WIDGET_LEVELS, 5, wd->TopEdge() + wdh + 5,
                    w/2 - 7, h - (wdh + 5) * 3 - 5, &levels,
                    -1, WIDGET_HSCROLLKEY|WIDGET_VSCROLLKEY,
                    NULL, this );
  levwidget->SetHook( this );

  mapwidget = new MapWidget( 0, w/2 + 2, levwidget->TopEdge(),
              w/2 - 7, levwidget->Height(), 0, this );
  maxmap = *mapwidget;

  wd = new ButtonWidget( IW_WIDGET_START, 5, h - wdh - 2, (w - 10) / 2 - 2, wdh,
                WIDGET_DEFAULT, MSG(MSG_B_START), this );
  wd->SetHook( this );

  wd = new ButtonWidget( GUI_QUIT, wd->LeftEdge() + wd->Width() + 4,
                wd->TopEdge(), wd->Width(), wdh,
                0, MSG(MSG_B_EXIT), this );

  Draw();
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::~InitWindow
// DESCRIPTION: Destroy window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

InitWindow::~InitWindow( void ) {
  TLWNode *n;
  initLevInfo *li;

  while ( !levels.IsEmpty() ) {
    n = static_cast<TLWNode *>( levels.RemHead() );
    li = (initLevInfo *)n->UserData();
    if ( li ) {
      delete li->full_path;
      delete li;
    }
    delete n;
  }

  while ( !saves.IsEmpty() ) {
    n = static_cast<TLWNode *>( saves.RemHead() );
    li = (initLevInfo *)n->UserData();
    if ( li ) {
      delete li->full_path;
      delete li;
    }
    delete n;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::Draw
// DESCRIPTION: Draw the main window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void InitWindow::Draw( void ) {
  Window::Draw();
  sfont->Write( MSG(MSG_OPTIONS), this, 5,
               gtypewidget->TopEdge() + gtypewidget->Height() + 9 );
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::WidgetActivated
// DESCRIPTION: Handle activation of widgets in the window.
// PARAMETERS : button - calling widget
//              win    - pointer to active window
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status InitWindow::WidgetActivated( Widget *button, Window *win ) {
  TLWNode *n;
  GUI_Status rc = GUI_OK;

  switch ( button->ID() ) {
  case IW_WIDGET_LEVELS:  // user selected mission, show map
    n = static_cast<TLWNode *>( levwidget->Selected() );
    if ( n ) {
      string *lname = ((initLevInfo *)n->UserData())->full_path;
      Mission *m = LoadMission( lname->c_str() );
      if ( m ) {
        Map &map = m->GetMap();
        unsigned char magnify = MIN( maxmap.Width() / map.Width(),
                                     maxmap.Height() / map.Height() );
        if ( magnify == 0 ) magnify = 1;
        else if ( magnify > 6 ) magnify = 6;

        mapwidget->w = magnify * map.Width() + 2;
        mapwidget->h = magnify * map.Height() + 2 + magnify/2;
        mapwidget->Center( maxmap );
        mapwidget->Clip( maxmap );

        mapwidget->SetPlayerColors( m->GetPlayer(PLAYER_ONE).LightColor(),
                                    m->GetPlayer(PLAYER_TWO).LightColor() );
        mapwidget->SetMap( &map, Rect(0,0,250,250), magnify );
        delete m;

        Draw();
        Show();
      }
    }
    break;
  case IW_WIDGET_START:
   /*If there is no level selected for loading, inform user */
    n = static_cast<TLWNode *>( levwidget->Selected() );
    if ( !n ) 
	    new NoteWindow( MSG(MSG_ERROR), MSG(MSG_ERR_NO_MAP), WIN_CLOSE_ESC, view );
    else {
  	/* Load the specified file */ 
    	string *lname = ((initLevInfo *)n->UserData())->full_path;      
    	Gam = new Game( view );
   	int val = Gam->Load( lname->c_str() );

	//Alway set game to local locale (grin)
	Gam->GetMission()->SetLocale( CFOptions.GetLanguage() );

	//val!=0 means unable to load
	if ( val!=0 ) {
        	NoteWindow *nw = new NoteWindow( MSG(MSG_ERROR), MSG(MSG_ERR_LOAD_MAP), 0, view );
	        nw->SetButtonID( 0, GUI_RESTART );
	} else {		
		/* Phew, File is loaded and correct */
		Player &p1 = Gam->GetMission()->GetPlayer( PLAYER_ONE );
		Player &p2 = Gam->GetMission()->GetPlayer( PLAYER_TWO );
		unsigned short mflags = Gam->GetMission()->GetFlags();	
		
		//if we aren't dealing with a saved file, build new settings
	        if (!(mflags & GI_SAVEFILE)) {
		     mflags &= ~(GI_PBEM|GI_AI|GI_AI_VS_AI); //insures not PBEM and AI
		     mflags |= ((CFOptions.IsPBEM() ? GI_PBEM : 0) | 
			      (CFOptions.IsAI_VS_AI() ? GI_AI_VS_AI : 0) |
			      (CFOptions.IsAI() ? GI_AI : 0) | GI_SAVEFILE); //GI_SAVEFILE prevents future mucking
		     //if its an AI game, prompt for AI setup
	  	     if (mflags & (GI_AI|GI_AI_VS_AI)) {
			AISelector *ais;
        		if (mflags & GI_AI_VS_AI) {
				p1.SetType( COMPUTER );
				ais = new AISelector(view, SELECT_BOTH_AI);
			} else 	ais = new AISelector(view, SELECT_P2_AI);
			p2.SetType( COMPUTER );
		      	ais->Show();
			int status = ais->EventLoop();
			ais->Close();
			if (status != GUI_CLOSE)
			{
			        delete Gam;
		      		Gam = NULL;
 				return (GUI_Status) status;
			}
			if (!ais->IsDone())
				return GUI_RESTART;
			p1.SetAIID(ais->GetPlayerOneAIID());
			p2.SetAIID(ais->GetPlayerTwoAIID());
	      	     }
	      	}
		// now we can respect 'mflags': they should be right 
		Gam->GetMission()->SetFlags(mflags);
			
		//get password
 	        const char *pass = Gam->GetMission()->GetPassword();
				
		//if (not locked/before first turn) or answer correctly, continue
		if ( !pass || Gam->CheckPassword( MSG(MSG_ACCESS_RESTRICTED), MSG(MSG_ENTER_CODE), pass, 0 ) ) 
		{
			//Give Birth to the AI's
			if (mflags & GI_AI_VS_AI) {
				p1.SetAI(AI::AIHelper(p1.GetAIID(), *(Gam->GetMission())));
				p2.SetAI(AI::AIHelper(p2.GetAIID(), *(Gam->GetMission())));
			} else if (mflags & GI_AI)
				p2.SetAI(AI::AIHelper(p2.GetAIID(), *(Gam->GetMission())));
			
			//if this is a saved game, it may have a history, 
		  	if ( !Gam->GetMission()->GetHistory() ) {
				History *history = new History();
				history->StartRecording( Gam->GetMission()->GetUnits() );
				Gam->GetMission()->SetHistory( history );
			}
			
			view->CloseWindow( this );
		        //wipe out the Emblem
			view->Flood( Color(CF_COLOR_SHADOW) );
        		view->Update();
	          	Gam->InitWindows();
        	  	rc = Gam->StartTurn();         // let the games begin...
		} else {
	          delete Gam;
        	  Gam = NULL;
	          new NoteWindow( MSG(MSG_ERR_NO_ACCESS), MSG(MSG_ERR_INVALID_CODE), WIN_CLOSE_ESC, view );
        	}
	} //end good load
      } //end no selected file
    break;
  case IW_WIDGET_NEW:
    newloadwidget->SetID( IW_WIDGET_LOAD );
    newloadwidget->SetTitle( MSG(MSG_B_LOAD_GAME) );
    newloadwidget->Draw();
    levwidget->SwitchList( &levels, -1 );
    gtypewidget->Enable();
    gtypewidget->Draw();
    Show();
    break;
  case IW_WIDGET_LOAD:
    newloadwidget->SetID( IW_WIDGET_NEW );
    newloadwidget->SetTitle( MSG(MSG_B_NEW_GAME) );
    newloadwidget->Draw();
    levwidget->SwitchList( &saves, -1 );
    gtypewidget->Disable();
    gtypewidget->Draw();
    Show();
    break;
  case IW_WIDGET_TYPE:
    CFOptions.SetGameType( static_cast<CycleWidget *>(button)->GetValue() );
    break;
  case IW_WIDGET_VIDEO:
    new VideoOptionsWindow( view );
    break;
  case IW_WIDGET_LANGUAGE:
    new LocaleOptionsWindow( 0, view );
    break;
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::CompleteFilesList
// DESCRIPTION: For the levels list display we need some information
//              about the maps (password protection, 1 or 2 players).
//              This information is attached to the list items.
// PARAMETERS : list - list of files created using create_files_list()
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void InitWindow::CompleteFilesList( TLWList &list ) const {
  initLevInfo *info;
  bool reorder = false;

  for ( TLWNode *n = static_cast<TLWNode *>(list.Head()), *next; n; n = next ) {
    next = static_cast<TLWNode *>(n->Next());
    string *fname = (string *)n->UserData();
    bool ok = false;

    Mission *m = LoadMission( fname->c_str() );
    if ( m ) {
      string name;

      // for new mission use mission title if present, for saved games file name
      if ( (m->GetFlags() & GI_SAVEFILE) || !m->GetName() ) {
        name.assign( n->Name() );
        name.erase( name.size() - 4 );    // remove .sav/.lev
      } else {
        name.assign( m->GetName() );
        reorder = true;
      }

      name += ' ';
      name += '(';
      if (m->GetFlags() & GI_AI_VS_AI)
	      name.append("AI-VS-AI");
      else if (m->GetFlags() & GI_PBEM)
	      name.append("Email Game");
      else name += ((m->GetFlags() & GI_AI) != 0 ? '1' : '2');
      name += ')';

      info = new initLevInfo;
      info->password = m->GetPassword() != 0;
      info->full_path = fname;
      n->SetName( name );
      n->SetUserData( info );
      delete m;
      ok = true;
    }

    if ( !ok ) {
      n->Remove();
      delete (string *)n->UserData();
      delete n;
    }
  }

  if ( reorder ) list.Sort();
}

////////////////////////////////////////////////////////////////////////
// NAME       : InitWindow::LoadMission
// DESCRIPTION: Load a mission into memory for inspection.
// PARAMETERS : filename - mission filename
// RETURNS    : mission if successful, NULL on error
////////////////////////////////////////////////////////////////////////

Mission *InitWindow::LoadMission( const char *filename ) const {
  Mission *m = 0;

  File file( filename );
  if ( file.Open( "rb" ) ) {

    m = new Mission();
    if ( m->Load( file ) != -1 ) {
      m->SetLocale( CFOptions.GetLanguage() );
    } else {
      delete m;
      m = 0;
    }
  }

  return m;
}

////////////////////////////////////////////////////////////////////////
// NAME       : TitleWindow::TitleWindow
// DESCRIPTION: Create a mostly invisible window containing the title
//              screen image.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

TitleWindow::TitleWindow( View *view ) : Window( WIN_CENTER, view ) {
  string tname = get_data_dir();
  append_path_delim( tname );
  tname.append( CF_TITLE_SCREEN );

  if ( LoadBMP( tname.c_str() ) ) SetSize( 0, 0, 0, 0 );
  else {
    DisplayFormat();
    Center( *view ); 

    Show();
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : TitleWindow::HandleEvent
// DESCRIPTION: Close (or rather, return control to main) when a key or
//              a mouse button is pressed.
// PARAMETERS : event - event received by the event handler
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status TitleWindow::HandleEvent( const SDL_Event &event ) {
  GUI_Status rc = GUI_OK;
  if ( (event.type == SDL_KEYDOWN) || (event.type == SDL_MOUSEBUTTONDOWN) )
    rc = GUI_CLOSE;
  return rc;
}


////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::VideoOptionsWindow
// DESCRIPTION: Show the video options menu.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

VideoOptionsWindow::VideoOptionsWindow( View *view ) :
     GenericOptionsWindow( MSG(MSG_OPTIONS_VIDEO), view ) {
  SDL_Rect std_sizes[] = { { 0, 0, 1280, 1024 }, { 0, 0, 1024, 768 },
                           { 0, 0, 800, 600}, { 0, 0, 640, 480 } };
  SDL_Rect *std_modes[5];
  short current = -1;

  // create modes list
  SDL_Rect **sdl_modes = SDL_ListModes( NULL, SDL_FULLSCREEN );

  // if any mode is ok offer some standard ones
  if ( sdl_modes == (SDL_Rect **)-1 ) {
    sdl_modes = std_modes;
    std_modes[0] = &std_sizes[0];
    std_modes[1] = &std_sizes[1];
    std_modes[2] = &std_sizes[2];
    std_modes[3] = &std_sizes[3];
    std_modes[4] = NULL;
  }
 
  if ( sdl_modes != NULL ) {
    for ( int i = 0; sdl_modes[i]; ++i ) AddMode( sdl_modes[i] );

    // add current mode
    SDL_Rect screen = { 0, 0, view->Width(), view->Height() };
    current = AddMode( &screen );
  }

  // set window size
  const char *fslabel = MSG(MSG_B_OPT_FULLSCREEN);
  unsigned short width = sfont->TextWidth(fslabel) + DEFAULT_CBW_SIZE + 20;
  if ( width < sfont->Width() * 15 ) width = sfont->Width() * 15;
  SetLayout( width, sfont->Height() * 11 + 30 );

  // create widgets
  const Rect &b = GetBounds();

  modewidget = new TextListWidget( 0, b.x + 5, b.y + 5,
                    b.w - 10, sfont->Height() * 10, &modes, current,
                    WIDGET_HSCROLLKEY|WIDGET_VSCROLLKEY|WIDGET_ALIGN_CENTER,
                    NULL, this );

  fswidget = new CheckboxWidget( 0, modewidget->LeftEdge() + 10,
                   modewidget->TopEdge() + modewidget->Height() + 10,
                   DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, view->IsFullScreen(),
                   WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
                   fslabel, this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::~VideoOptionsWindow
// DESCRIPTION: Destroy the video options window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

VideoOptionsWindow::~VideoOptionsWindow( void ) {
  TLWNode *n2;

  for ( TLWNode *n = static_cast<TLWNode *>(modes.Head()); n; n = n2 ) {
    n2 = static_cast<TLWNode *>(n->Next());
    n->Remove();

    SDL_Rect *rect = (SDL_Rect *)n->UserData();
    delete rect;
    delete n;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::AddMode
// DESCRIPTION: Add another resultion to the list. Duplicates and
//              invalid sizes will be rejected.
// PARAMETERS : res - resolution to add
// RETURNS    : position at which the mode has been added (or existed
//              before); -1 if mode was rejected
////////////////////////////////////////////////////////////////////////

short VideoOptionsWindow::AddMode( SDL_Rect *res ) {
  short rc = -1;

  if ( (res->w >= MIN_XRES) && (res->h >= MIN_YRES) ) {
    TLWNode *walk, *prev = NULL;
    bool add = true;
    rc = 0;

    for ( walk = static_cast<TLWNode *>(modes.Head());
          walk; prev = walk, walk = static_cast<TLWNode *>(walk->Next()) ) {
      SDL_Rect *nres = (SDL_Rect *)walk->UserData();
      if ( nres->w <= res->w ) {
        if ( (nres->w == res->w) && (nres->h == res->h) ) add = false;
        break;
      }
      ++rc;
    }

    if ( add ) {
      char buf[16];
      sprintf( buf, "%d x %d", res->w, res->h );

      SDL_Rect *mode = new SDL_Rect;
      mode->w = res->w;
      mode->h = res->h;
      TLWNode *n = new TLWNode( buf, mode );
      modes.InsertNode( n, prev );
    }
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : VideoOptionsWindow::WidgetActivated
// DESCRIPTION: When the user pushes the 'OK' button, switch to the
//              selected video mode.
// PARAMETERS : button - calling widget
//              win    - pointer to active window
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status VideoOptionsWindow::WidgetActivated( Widget *button, Window *win ) {
  view->CloseWindow( win );

  TLWNode *mode = static_cast<TLWNode *>( modewidget->Selected() );

  if ( mode ) {
    bool fs = fswidget->Clicked();
    unsigned long mode_flags = SDL_HWSURFACE|(fs ? SDL_FULLSCREEN : 0);

    // if selected mode is the same as current mode only check for fullscreen
    // dimensions of the selected mode are available in the user_data field
    SDL_Rect *dim = (SDL_Rect *)mode->UserData();
    if ( (dim->w == view->Width()) && (dim->h == view->Height()) ) {
      if ( fs != view->IsFullScreen() ) view->ToggleFullScreen();
    } else {
      view->SetVideoMode( dim->w, dim->h, DISPLAY_BPP, mode_flags );
    }
  }
  return GUI_OK;
}


////////////////////////////////////////////////////////////////////////
// NAME       : GenericOptionsWindow::SetLayout
// DESCRIPTION: Set the size of the window. This must be called before
//              trying to display it or using GetBounds().
// PARAMETERS : w     - width (only the part needed by the subclass)
//              h     - height (only the part needed by the subclass)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void GenericOptionsWindow::SetLayout( unsigned short w, unsigned short h ) {
  unsigned short tw = lfont->TextWidth(title);

  if ( w < tw + 20 ) w = tw + 20;

  clientarea = Rect( 5, 13 + lfont->Height(), w, h );

  Window::SetSize( w + 10, h + lfont->Height() + sfont->Height() + 25 );

  Widget *wd = new ButtonWidget( B_ID_OK, 1, Height() - sfont->Height() - 9,
                   Width() - 2, sfont->Height() + 8, WIDGET_DEFAULT, MSG(MSG_B_OK), this );
  wd->SetHook( this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : GenericOptionsWindow::Draw
// DESCRIPTION: Draw the options window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void GenericOptionsWindow::Draw( void ) {
  Window::Draw();

  DrawBox( clientarea, BOX_RECESSED );

  int xpos = (w - lfont->TextWidth(title)) / 2;
  lfont->Write( title, this, xpos + 3, 8, view->GetBGPen() );
  lfont->Write( title, this, xpos, 5, view->GetFGPen() );
}


////////////////////////////////////////////////////////////////////////
// NAME       : GeneralOptionsWindow::GeneralOptionsWindow
// DESCRIPTION: Show the general options menu.
// PARAMETERS : mv   - current map view
//              view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

GeneralOptionsWindow::GeneralOptionsWindow( MapView &mv, View *view ) :
     GenericOptionsWindow( MSG(MSG_OPTIONS_GENERAL), view ), mv(mv) {
  unsigned short widths[3] = { 25 + DEFAULT_CBW_SIZE,
                               25 + DEFAULT_CBW_SIZE,
                               35 + DEFAULT_CBW_SIZE }, maxw = 0;
  const char *labels[3];

  labels[0] = MSG(MSG_B_OPT_DAMAGE);
  labels[1] = MSG(MSG_B_OPT_REPLAYS);
  labels[2] = MSG(MSG_B_OPT_REPLAYS_QUICK);

  for ( int i = 0; i < 3; ++i ) {
    widths[i] += sfont->TextWidth(labels[i]);
    if ( widths[i] > maxw ) maxw = widths[i];
  }

  SetLayout( maxw, sfont->Height() * 3 + 35 );

  // create widgets
  const Rect &b = GetBounds();

  // damage indicator
  diwidget = new CheckboxWidget( 0, b.x + 5, b.y + 5,
               DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, mv.UnitStatsEnabled(),
               WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
               labels[0], this );

  // replay
  repwidget = new CheckboxWidget( 0, b.x + 5, b.y + diwidget->Height() + 10,
              DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, CFOptions.GetTurnReplay(),
              WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
              labels[1], this );
  repwidget->SetHook(this);

  // quick replays
  qrepwidget = new CheckboxWidget( 0, b.x + 15, repwidget->y + repwidget->Height() + 5,
              DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, CFOptions.GetQuickReplay(),
              WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT|
              (repwidget->Clicked() ? 0 : WIDGET_DISABLED),
              labels[2], this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : GeneralOptionsWindow::WidgetActivated
// DESCRIPTION: When the user pushes the 'OK' button, propagate the
//              selected options.
// PARAMETERS : widget - calling widget
//              win    - pointer to active window
// RETURNS    : GUI_OK
////////////////////////////////////////////////////////////////////////

GUI_Status GeneralOptionsWindow::WidgetActivated( Widget *widget, Window *win ) {

  if ( widget == repwidget ) {

    if ( repwidget->Clicked() ) qrepwidget->Enable();
    else {
      qrepwidget->Release();
      qrepwidget->Disable();
    }

    Draw();
    Show();

  } else {

    view->CloseWindow( win );

    if ( mv.UnitStatsEnabled() != diwidget->Clicked() ) {
      CFOptions.SetDamageIndicator( diwidget->Clicked() );

      if ( diwidget->Clicked() ) mv.EnableUnitStats();
      else mv.DisableUnitStats();

      // now redraw the map display
      mv.Draw();
      view->Refresh( mv );
    }

    CFOptions.SetTurnReplay( repwidget->Clicked() );
    CFOptions.SetQuickReplay( qrepwidget->Clicked() );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : LocaleOptionsWindow::LocaleOptionsWindow
// DESCRIPTION: Show the language options menu.
// PARAMETERS : game - current game. If it is NULL we assume the window
//                     has been opened from the initial start window.
//                     Otherwise we do an in-game language switch.
//              view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

LocaleOptionsWindow::LocaleOptionsWindow( Game *game, View *view ) :
     GenericOptionsWindow( MSG(MSG_OPTIONS_LANGUAGE), view ), game(game) {

  // collect language data files
  short current = ReadLocales();
  short found = locales.CountNodes();
  if ( found > 5 ) found = 5;

  SetLayout( sfont->Width() * 20 + 20, found * (sfont->Height() + 2) + 25 );

  // create widgets
  const Rect &b = GetBounds();

  locwidget = new TextListWidget( 0, b.x + 5, b.y + 5, b.w - 10, b.h - 10,
                  &locales, current,
                  WIDGET_HSCROLLKEY|WIDGET_VSCROLLKEY|WIDGET_ALIGN_CENTER,
                  NULL, this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : LocaleOptionsWindow::~LocaleOptionsWindow
// DESCRIPTION: Free resources.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

LocaleOptionsWindow::~LocaleOptionsWindow( void ) {
  while ( !locales.IsEmpty() ) {
    TLWNode *n = static_cast<TLWNode *>(locales.RemHead());
    delete (Language *)n->UserData();
    delete n;
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : LocaleOptionsWindow::ReadLocales
// DESCRIPTION: Get the list of locales.
// PARAMETERS : -
// RETURNS    : index of currently selected language
////////////////////////////////////////////////////////////////////////

short LocaleOptionsWindow::ReadLocales( void ) {

  TLWList files;
  FileWindow::CreateFilesList( get_locale_dir().c_str(),
                               ".dat", files );
  short i = 0, current = -1;

  while ( !files.IsEmpty() ) {
    TLWNode *n = static_cast<TLWNode *>(files.RemHead());

    string *fname = (string *)n->UserData();

    Language *l = new Language();
    if ( l->ReadCatalog( fname->c_str() ) != -1 ) {
      n->SetName( l->Name() );
      n->SetUserData( l );
      n->SetID( i );

      locales.AddTail( n );

      if ( string(l->ID()) == CFOptions.GetLanguage() )
        current = i;

      ++i;
    } else {
      delete l;
      delete n;
    }

    delete fname;
  }

  return current;
}

////////////////////////////////////////////////////////////////////////
// NAME       : LocaleOptionsWindow::WidgetActivated
// DESCRIPTION: When the user pushes the 'OK' button, propagate the
//              selected options.
// PARAMETERS : widget - calling widget (only OK)
//              win    - pointer to active window
// RETURNS    : GUI_OK
////////////////////////////////////////////////////////////////////////

GUI_Status LocaleOptionsWindow::WidgetActivated( Widget *widget, Window *win ) {
  GUI_Status rc = GUI_OK;

  view->CloseWindow( win );

  TLWNode *n = static_cast<TLWNode *>(locwidget->Selected());
  if ( n ) {
    Language *l = (Language *)n->UserData();
    if ( string(l->ID()) != CFOptions.GetLanguage() ) {

      Lang = *l;
      CFOptions.SetLanguage( l->ID() );

      if ( game ) {
        // we've changed the language in-game:
        // simply set the language and update the panel
        game->InitKeys();
        game->GetMission()->SetLocale( l->ID() );

        MapWindow *mwin = game->GetMapWindow();
        MapView *mv = mwin->GetMapView();
        mwin->GetPanel()->Update( mv->GetMap()->GetMapObject(mv->Cursor()) );
      } else {
        // we've been called from the start menu: set the new language
        // and start over. This way we don't need to update the individual
        // widgets
        rc = GUI_RESTART;
      }
    }
  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundOptionsWindow::SoundOptionsWindow
// DESCRIPTION: Show the sound options menu.
// PARAMETERS : view - view to attach the window to
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

#ifndef DISABLE_SOUND

const short SoundOptionsWindow::B_ID_SFX       = 10;
const short SoundOptionsWindow::B_ID_MUSIC     = 11;
const short SoundOptionsWindow::S_ID_VOL_SFX   = 12;
const short SoundOptionsWindow::S_ID_VOL_MUSIC = 13;

SoundOptionsWindow::SoundOptionsWindow( View *view ) :
     GenericOptionsWindow( MSG(MSG_OPTIONS_AUDIO), view ),
     volgfx( view->GetSystemIcons(), 145, 46, 12 ,12 ) {
  const char *sfxlabel = MSG(MSG_B_OPT_SFX);
  const char *musiclabel = MSG(MSG_B_OPT_MUSIC);
  unsigned short sfxw = sfont->TextWidth(sfxlabel);
  unsigned short musw = sfont->TextWidth(musiclabel);

  SetLayout( MAX(sfxw, musw) + DEFAULT_CBW_SIZE +
             volgfx.Width() + 100,
             view->SmallFont()->Height() * 2 + 20 );

  // create widgets
  const Rect &b = GetBounds();
  Widget *wd;

  // sfx
  wd = new CheckboxWidget( B_ID_SFX, b.x + 5, b.y + 5,
           DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, Audio::GetSfxState(),
           WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,
           sfxlabel, this );
  wd->SetHook( this );

  // sfx volume
  short xoff = wd->LeftEdge() + wd->Width() + MAX(sfxw, musw) +
               volgfx.Width() + 15;
  sfxvol = new SliderWidget( S_ID_VOL_SFX,
           xoff, wd->TopEdge() + (wd->Height() - DEFAULT_SLIDER_SIZE)/2,
           b.w - xoff - 5, DEFAULT_SLIDER_SIZE, 0, MIX_MAX_VOLUME,
           Audio::GetSfxVolume(), 20,
           WIDGET_HSCROLL|WIDGET_HSCROLLKEY|
             (Audio::GetSfxState() ? 0 : WIDGET_DISABLED),
           NULL, this );
  sfxvol->SetHook( this );

  // music
  wd = new CheckboxWidget( B_ID_MUSIC,
           wd->LeftEdge(), wd->TopEdge() + wd->Height() + 5,
           DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE, Audio::GetMusicState(),
	   WIDGET_STYLE_NOBORDER|WIDGET_STYLE_GFX|WIDGET_ALIGN_RIGHT,  
           musiclabel, this );
  wd->SetHook( this );

  // music volume
  musicvol = new SliderWidget( S_ID_VOL_MUSIC,
           xoff, wd->TopEdge() + (wd->Height() - DEFAULT_SLIDER_SIZE)/2,
           b.w - xoff - 5, DEFAULT_SLIDER_SIZE, 0, MIX_MAX_VOLUME,
           Audio::GetMusicVolume(), 20, WIDGET_HSCROLL|WIDGET_VSCROLLKEY|
	     (Audio::GetMusicState() ? 0 : WIDGET_DISABLED),
           NULL, this );
  musicvol->SetHook( this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundOptionsWindow::WidgetActivated
// DESCRIPTION: When the user activates a widget, propagate the changes
//              to the sound layer.
// PARAMETERS : widget - calling widget
//              win    - pointer to active window
// RETURNS    : GUI_OK
////////////////////////////////////////////////////////////////////////

GUI_Status SoundOptionsWindow::WidgetActivated( Widget *widget, Window *win ) {

  switch ( widget->ID() ) {
  case B_ID_SFX:
    Audio::ToggleSfxState();

    if ( Audio::GetSfxState() ) sfxvol->Enable();
    else sfxvol->Disable();
    break;

  case S_ID_VOL_SFX:
    Audio::SetSfxVolume( sfxvol->Level() );
    break;

  case B_ID_MUSIC:
    Audio::ToggleMusicState();

    if ( Audio::GetMusicState() ) musicvol->Enable();
    else musicvol->Disable();
    break;

  case S_ID_VOL_MUSIC:
    Audio::SetMusicVolume( musicvol->Level() );
    break;

  case B_ID_OK:
    view->CloseWindow( this );
    break;

  default:
    break;
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : SoundOptionsWindow::Draw
// DESCRIPTION: Draw the sound options window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void SoundOptionsWindow::Draw( void ) {
  GenericOptionsWindow::Draw();

  short xoff = sfxvol->LeftEdge() - volgfx.Width() - 5;

  volgfx.Draw( this, xoff,
         sfxvol->TopEdge() + (sfxvol->Height() - volgfx.Height())/2 );
  volgfx.Draw( this, xoff,
         musicvol->TopEdge() + (musicvol->Height() - volgfx.Height())/2 );
}
#endif

