// CoMET - The Crimson Fields Map Editing Tool
// Copyright (C) 2002-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// edwindow.cpp
////////////////////////////////////////////////////////////////////////

#include "edwindow.h"
#include "extwindow.h"
#include "extwindow2.h"
#include "filewindow.h"
#include "gfxwidget.h"
#include "fileio.h"
#include "main.h"

// button IDs for the context menu
#define B_ID_MENU_BASE       200

// inititalize static vars
const short EdWindow::L_ID_UNITS      = 0;
const short EdWindow::L_ID_TILES      = 1;
const short EdWindow::S_ID_VERT       = 2;
const short EdWindow::S_ID_HORIZ      = 3;

const short EdWindow::B_ID_TILE_GRAB  = B_ID_MENU_BASE + 0;
const short EdWindow::B_ID_TILE_SWAP  = B_ID_MENU_BASE + 1;
const short EdWindow::B_ID_UNIT_INFO  = B_ID_MENU_BASE + 2;
const short EdWindow::B_ID_UNIT_EDIT  = B_ID_MENU_BASE + 3;
const short EdWindow::B_ID_UNIT_DEL   = B_ID_MENU_BASE + 4;
const short EdWindow::B_ID_BLD_CREATE = B_ID_MENU_BASE + 5;
const short EdWindow::B_ID_BLD_EDIT   = B_ID_MENU_BASE + 6;
const short EdWindow::B_ID_BLD_DEL    = B_ID_MENU_BASE + 7;
const short EdWindow::B_ID_EVENTS     = B_ID_MENU_BASE + 8;
const short EdWindow::B_ID_NEW        = B_ID_MENU_BASE + 9;
const short EdWindow::B_ID_LOAD       = B_ID_MENU_BASE + 10;
const short EdWindow::B_ID_SAVE       = B_ID_MENU_BASE + 11;
const short EdWindow::B_ID_EXPORT     = B_ID_MENU_BASE + 12;
const short EdWindow::B_ID_VALIDATE   = B_ID_MENU_BASE + 13;
const short EdWindow::B_ID_SETTINGS   = B_ID_MENU_BASE + 14;
const short EdWindow::B_ID_QUIT       = B_ID_MENU_BASE + 15;

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::EdWindow
// DESCRIPTION: Create the editing window.
// PARAMETERS : mapdir - directory containing the map files
//              view   - pointer to the view to use
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdWindow::EdWindow( const string &mapdir, View *view ) :
          Window( view->x, view->y, view->w, view->h, 0, view ),
          panel( view->w - GFX_WIDTH * 4 - DEFAULT_SLIDER_SIZE - 20,
                 view->y, GFX_WIDTH * 4 + DEFAULT_SLIDER_SIZE + 20, view->h ),
          mv( this,
              Rect( DEFAULT_SLIDER_SIZE, 0,
              w - panel.w - DEFAULT_SLIDER_SIZE, h - DEFAULT_SLIDER_SIZE),
              MV_DISABLE|MV_DISABLE_FOG ), mapdir(mapdir) {
  mode = ED_MODE_VIEW;
  mission = NULL;
  selected = -1;

  unsigned short height = MIN( GFX_HEIGHT * 6 + 4, h / 3 - 10 );
  unit_wd = new UnitWidget( L_ID_UNITS, panel.x + 5,
                            panel.y + panel.h - 6 - height,
                            panel.w - 10, height, 0, NULL,
                            this, mv, NULL, GFX_WIDTH, GFX_HEIGHT );
  unit_wd->SetHook( this );

  tile_wd = new TileWidget( L_ID_TILES, unit_wd->x,
                            panel.y + sfont->Height() + 15,
                            unit_wd->w, panel.h - height - sfont->Height() - 30,
                            0, NULL, this, mv, NULL, GFX_WIDTH, GFX_HEIGHT );
  tile_wd->SetHook( this );

  ud_wd = new SliderWidget( S_ID_VERT, 0, 0, DEFAULT_SLIDER_SIZE, mv.Height(),
                            0, 0, 0, 1, WIDGET_VSCROLL, NULL, this );
  ud_wd->SetHook( this );
  lr_wd = new SliderWidget( S_ID_HORIZ, DEFAULT_SLIDER_SIZE, h - DEFAULT_SLIDER_SIZE,
                            mv.Width(), DEFAULT_SLIDER_SIZE,
                            0, 0, 0, 1, WIDGET_HSCROLL, NULL, this );
  lr_wd->SetHook( this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::~EdWindow
// DESCRIPTION: Delete the editing window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdWindow::~EdWindow( void ) {
  delete mission;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::Draw
// DESCRIPTION: Draw the editing window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::Draw( void ) {
  mv.Draw();
  DrawBack( panel );
  DrawBox( panel, BOX_RAISED );

  Widget *wd = widgets;
  while ( wd ) {
    if ( !wd->Hidden() ) wd->Draw();
    wd = wd->next;
  }

  PrintCursorPos();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::PrintCursorPos
// DESCRIPTION: Print current cursor position to the panel.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::PrintCursorPos( void ) {
  Rect coord( panel.x + 5, panel.y + 5, panel.w - 10, sfont->Height() );
  char posbuf[16];
  DrawBack( coord );

  if ( mission ) {
    sprintf( posbuf, "X/Y: %d/%d", mv.Cursor().x, mv.Cursor().y );
    sfont->Write( posbuf, this, coord.x, coord.y );
  }
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::HandleEvent
// DESCRIPTION: React on user input.
// PARAMETERS : event - event received by the event handler
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdWindow::HandleEvent( const SDL_Event &event ) {
  GUI_Status rc = Window::HandleEvent( event );

  if ( rc == GUI_OK ) {

    // check for keyboard commands
    if ( event.type == SDL_KEYDOWN ) {

      if ( !mission )
        ShowContextMenu( Point(-1, -1) );
      else {

        switch ( event.key.keysym.sym ) {
        case SDLK_KP1: case SDLK_KP2: case SDLK_KP3:
        case SDLK_KP4: case SDLK_KP6: case SDLK_KP7:
        case SDLK_KP8: case SDLK_KP9:
        case SDLK_LEFT: case SDLK_RIGHT: case SDLK_UP: case SDLK_DOWN:
          MoveCursor( event.key.keysym.sym );
          break;
        case SDLK_SPACE:      // equivalent to left mouse button
          LeftMouseButton( mv.Cursor() );
          break;
        case SDLK_q:
          Quit();
          break;
        case SDLK_ESCAPE:
          tile_wd->Select( -1 );
          unit_wd->Select( -1 );
          break;
        case SDLK_RETURN:
          ShowContextMenu( mv.Cursor() );

        case SDLK_u: {		// edit unit
          Unit *u = mv.GetMap()->GetUnit( mv.Cursor() );
          if ( u ) {
            EdUnitWindow *euw = new EdUnitWindow( *u, *mission, view );
            euw->SetMapView( mv );
          }
          break; }
        case SDLK_b: {		// edit building
          Building *b = mv.GetMap()->GetBuilding( mv.Cursor() );
          if ( b ) new EdBuildingWindow( *b, *mission, view );
          break; }

        default:
          break;
        }
      }

    } else if ( event.type == SDL_MOUSEBUTTONDOWN ) {
      if ( !mission )
        ShowContextMenu( Point(event.button.x - x, event.button.y - y) );
      else {
        Point pos;
        bool validhex = !mv.Pixel2Hex( event.button.x - x, event.button.y - y, pos );

        if ( event.button.button == SDL_BUTTON_LEFT ) {
          if ( validhex ) LeftMouseButton( pos );
        } else if ( event.button.button == SDL_BUTTON_RIGHT ) {
           ShowContextMenu( validhex ? pos : Point(-1,-1) );
        }
      }
    }
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::MoveCursor
// DESCRIPTION: Move the cursor in reaction to a key event.
// PARAMETERS : key - the key code used to give the order
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::MoveCursor( int key ) {
  Direction dir;

  switch ( key ) {
  case SDLK_KP1:  dir = SOUTHWEST; break;
  case SDLK_DOWN:
  case SDLK_KP2:  dir = SOUTH;     break;
  case SDLK_KP3:  dir = SOUTHEAST; break;
  case SDLK_KP7:  dir = NORTHWEST; break;
  case SDLK_UP:
  case SDLK_KP8:  dir = NORTH;     break;
  case SDLK_KP9:  dir = NORTHEAST; break;
  case SDLK_LEFT:
  case SDLK_KP4:  dir = WEST;      break;
  case SDLK_RIGHT:
  case SDLK_KP6:  dir = EAST;      break;
  default: return;
  }

  Point dest;
  if ( !mv.GetMap()->Dir2Hex( mv.Cursor(), dir, dest ) ) SetCursor( dest );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::SetCursor
// DESCRIPTION: Set the cursor to a new hex on the map. Contrary to the
//              low-level function in MapView this updates the display
//              at the old and new position if necessary.
// PARAMETERS : cursor - new cursor position
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::SetCursor( const Point &cursor ) {
  Rect upd;

  if ( mv.CursorEnabled() ) {
    upd = mv.SetCursor( Point(-1,-1) );  // disable cursor for hex update
    Show( upd );                         // update previous cursor position
  }

  upd = mv.SetCursor( cursor );
  Show( upd );

  PrintCursorPos();
  Show( panel );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::LeftMouseButton
// DESCRIPTION: Left mouse button has been pressed. See what we can do.
// PARAMETERS : hex - hex the button was clicked over
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::LeftMouseButton( const Point &hex ) {
  Map *map = mv.GetMap();

  if ( mode == ED_MODE_TERRAIN ) map->SetHexType( hex, selected );
  else if ( mode == ED_MODE_UNIT ) {
    if ( map->GetMapObject(hex) ) new NoteWindow( "Error!", "Hex occupied.", 0, view );
    else {
      UnitSet *us = map->GetUnitSet();
      const UnitType *ut = us->GetUnitInfo( selected < us->NumUT() ?
                                            selected : selected - us->NumUT() );
      mission->CreateUnit( ut, selected < us->NumUT() ? PLAYER_ONE : PLAYER_TWO, hex );
    }
  }

  SetCursor( hex );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::LoadMission
// DESCRIPTION: Load a mission from file and attach it to the display.
// PARAMETERS : file    - name of mission data file
//              filereq - pop up file requester
// RETURNS    : pointer to Mission object on success, NULL on error
////////////////////////////////////////////////////////////////////////

Mission *EdWindow::LoadMission( const char *file, bool filereq ) {
  string fn;

  if ( filereq )
    fn = GetFileName( file, ".lev", mapdir, WIN_FILE_LOAD );
  else fn = file;

  if ( fn.length() > 0 ) {
    delete mission;
    mv.Disable();

    mission = new Mission();
    if ( mission ) {
      int err = mission->Load( fn.c_str() );
      if ( err ) {
        delete mission;
        mission = NULL;
        new NoteWindow( "Error", "Couldn't load map. This could be a file type mismatch.", 0, view );
      } else {
        bool allowed = true;

        if ( mission->GetPassword() ) {
          PasswordWindow *pw = new PasswordWindow( "Access restricted", "Please enter access code",
                                   mission->GetPassword(), view );
          pw->EventLoop();
          allowed = pw->PasswordOk();
          view->CloseWindow( pw );
        }

        if ( allowed ) {
          Map &map = mission->GetMap();
          mode = ED_MODE_VIEW;
          mv.SetMap( &map );
          mv.SetCursor( Point(0,0) );

          tile_wd->SetTerrainSet( map.GetTerrainSet(), GFX_WIDTH, GFX_HEIGHT );
          unit_wd->SetUnitSet( map.GetUnitSet(), GFX_WIDTH, GFX_HEIGHT );

          short max = MAX( map.Height() * GFX_HEIGHT + GFX_OVERLAP_Y - mv.Height(), 0 );
          ud_wd->Adjust( 0, max, mv.Height() );
          ud_wd->ScrollTo( 0 );

          max = MAX( map.Width() * (GFX_WIDTH - GFX_OVERLAP_X) + GFX_OVERLAP_X - mv.Width(), 0 );
          lr_wd->Adjust( 0, max, mv.Width() );
          lr_wd->ScrollTo( 0 );

          mv.Enable();
        } else {
          Audio::PlaySfx( Audio::SND_GUI_ERROR, 0 );
          new NoteWindow( "Error", "Code rejected. Access denied.", WIN_CENTER, view );

          tile_wd->SetTerrainSet( NULL, GFX_WIDTH, GFX_HEIGHT );
          unit_wd->SetUnitSet( NULL, GFX_WIDTH, GFX_HEIGHT );
          delete mission;
          mission = NULL;
        }
      
        Draw();
        Show();
      }
    }
  }

  return mission;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::SaveMission
// DESCRIPTION: Save a mission to file.
// PARAMETERS : file    - name of mission data file
//              filereq - pop up file requester
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdWindow::SaveMission( const char *file, bool filereq ) {
  string fn;
  GUI_Status rc = GUI_OK;

  if ( filereq )
    fn = GetFileName( file, ".lev", mapdir, WIN_FILE_SAVE );
  else fn = file;

  if ( fn.length() > 0 ) {
    if ( mission->Save( fn.c_str() ) != 0 ) {
      new NoteWindow( "Error", "Could not save mission!", WIN_CENTER, view );
    }
  }
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::ExportMission
// DESCRIPTION: Save a mission to a plain text file.
// PARAMETERS : file    - default name of mission data file
//              filereq - pop up file requester
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdWindow::ExportMission( const char *file, bool filereq ) {
  string fn;
  GUI_Status rc = GUI_OK;

  if ( filereq )
    fn = GetFileName( file, ".src", mapdir, WIN_FILE_SAVE );
  else fn = file;

  if ( fn.length() > 0 ) mission->Export( fn.c_str() );
  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::NewMission
// DESCRIPTION: Create a new mission.
// PARAMETERS : size - map size
//              tset - tile set name
//              uset - unit set name
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdWindow::NewMission( const Point &size, const string &tset,
                                 const string &uset ) {
  TerrainSet *ts = new TerrainSet;
  UnitSet *us = new UnitSet;

  string tname( get_data_dir() + tset + ".tiles" );
  File tfile( tname );

  string uname( get_data_dir() + uset + ".units" );
  File ufile( uname );

  if ( !tfile.Open("rb") || ts->Load( tfile, tset.c_str() ) ) {
    new NoteWindow( "Error", "Tile set not available", 0, view );
    delete ts;
  } else if ( !ufile.Open("rb") || us->Load( ufile, uset.c_str() ) ) {
    new NoteWindow( "Error", "Unit set not available", 0, view );
    delete ts;
    delete us;
  } else {
    delete mission;

    mv.Disable();
    mode = ED_MODE_VIEW;

    mission = new Mission( size, ts, us );
    Map &map = mission->GetMap();
    mv.SetMap( &map );
    mv.SetCursor( Point(0,0) );

    tile_wd->SetTerrainSet( ts, GFX_WIDTH, GFX_HEIGHT );
    unit_wd->SetUnitSet( us, GFX_WIDTH, GFX_HEIGHT );

    short max = MAX( map.Height() * GFX_HEIGHT + GFX_OVERLAP_Y - mv.Height(), 0 );
    ud_wd->Adjust( 0, max, mv.Height() );
    ud_wd->ScrollTo( 0 );

    max = MAX( map.Width() * (GFX_WIDTH - GFX_OVERLAP_X) + GFX_OVERLAP_X - mv.Width(), 0 );
    lr_wd->Adjust( 0, max, mv.Width() );
    lr_wd->ScrollTo( 0 );

    mv.Enable();
    Draw();
    Show();
  }

  tfile.Close();
  ufile.Close();
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::GetFileName
// DESCRIPTION: Get the name of a file to load or to save to. Pop up a
//              file requester.
// PARAMETERS : filename - default filename
//              suffix   - filename suffix
//              dirname  - name of directory to look in
//              flag     - WIN_FILE_LOAD or WIN_FILE_SAVE
// RETURNS    : file name or empty string on error
////////////////////////////////////////////////////////////////////////

string EdWindow::GetFileName( const char *filename, const char *suffix,
                              const string &dirname, int flag ) const {
  string file( filename );
  if ( file.length() > 0 ) file.append( suffix );

  bool filesel;
  do {
    GUI_Status rc;
    DialogWindow *dw;
    bool done = false;
    FileWindow *fw = new FileWindow( dirname.c_str(), file.c_str(),
                                     suffix, flag, view );
    fw->ok->SetID( 1 );
    fw->cancel->SetID( 0 );

    do {
      filesel = false;
      rc = fw->EventLoop();

      if ( rc == 1 ) {
        file = fw->GetFile();
        if ( file.length() != 0 ) {
          view->CloseWindow( fw );
          done = true;
        }
      } else if ( rc == 0 ) {
        view->CloseWindow( fw );
        file.assign( "" );
        done = true;
      }

    } while ( !done );

    if ( file.length() > 0 && (flag == WIN_FILE_SAVE) && File::Exists( file ) ) {
      // if file exists let user confirm the write
      char *conmsg = new char[ file.length() + 20 ];
      strcpy( conmsg, file.c_str() );
      strcat( conmsg, " exists. Overwrite?" );
      dw = new DialogWindow( NULL, conmsg, "_Yes|_No", 1, 0, view );
      dw->SetButtonID( 0, 1 );
      dw->SetButtonID( 1, 0 );
      rc = dw->EventLoop();
      view->CloseWindow( dw );
      if ( rc == 0 ) {
        filesel = true;
        file = file_part( file );
      }
      delete [] conmsg;
    }

  } while ( filesel );

  return file;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::WidgetActivated
// DESCRIPTION: This method is called when the user selects one of the
//              widgets associated with this window - the tiles or units
//              list, the map scrollers, or one of a number of buttons
//              including the main menu.
// PARAMETERS : widget - pointer to calling widget
//              win    - parent window of calling widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdWindow::WidgetActivated( Widget *widget, Window *win ) {
  GUI_Status rc = GUI_OK;

  if ( win == this ) {

    switch ( widget->ID() ) {
    case L_ID_UNITS:
      if ( unit_wd->Selected() == -1 ) {
        if ( tile_wd->Selected() == -1 ) {
          mode = ED_MODE_VIEW;
          selected = -1;
        }
      } else {
        mode = ED_MODE_UNIT;
        selected = unit_wd->Selected();
        tile_wd->Select( -1 );
      }
      break;
    case L_ID_TILES:
      if ( tile_wd->Selected() == -1 ) {
        if ( unit_wd->Selected() == -1 ) {
          mode = ED_MODE_VIEW;
          selected = -1;
        }
      } else {
        mode = ED_MODE_TERRAIN;
        selected = tile_wd->Selected();
        unit_wd->Select( -1 );
      }
      break;

    case S_ID_VERT: {
      Point offset = mv.GetOffsets();
      mv.Scroll( 0, ud_wd->Level() - offset.y );
      Show( mv );
      break; }
    case S_ID_HORIZ: {
      Point offset = mv.GetOffsets();
      mv.Scroll( lr_wd->Level() - offset.x, 0 );
      Show( mv );
      break; }
    }

  } else {

    Map *map = mv.GetMap();

    if ( widget->ID() >= B_ID_MENU_BASE )
      static_cast<MenuWindow *>(win)->CloseParent();
    view->CloseWindow(win);

    switch ( widget->ID() ) {
    case B_ID_NEW: {
      NewMissionWindow *nmw = new NewMissionWindow( view );
      nmw->SetHook( this );
      break; }
    case B_ID_LOAD:
      LoadMission( "", true );
      break;
    case B_ID_SAVE:
      rc = SaveMission( mission->GetTitle().c_str(), true );
      break;
    case B_ID_EXPORT:
      rc = ExportMission( mission->GetTitle().c_str(), true );
      break;
    case B_ID_VALIDATE:
      ValidateMission();
      break;
    case B_ID_SETTINGS:
      new EdMissionSetupWindow( *mission, view );
      break;
    case B_ID_QUIT:
      Quit();
      break;

    case B_ID_TILE_GRAB:
      tile_wd->Select( map->HexTypeID(selected_hex) );
      break;
    case B_ID_TILE_SWAP:
      SwapTiles( map->HexTypeID(selected_hex), selected );
      break;

    case B_ID_UNIT_INFO:
      break;
    case B_ID_UNIT_EDIT: {
      EdUnitWindow *euw = new EdUnitWindow( *(map->GetUnit( selected_hex )), *mission, view );
      euw->SetMapView( mv );
      break; }
    case B_ID_UNIT_DEL:
      mission->DeleteUnit( map->GetUnit(selected_hex) );
      Show( mv.UpdateHex( selected_hex ) );
      break;

    case B_ID_BLD_CREATE:
      mission->CreateBuilding( PLAYER_ONE, selected_hex );  // fall through
    case B_ID_BLD_EDIT:
      new EdBuildingWindow( *(map->GetBuilding( selected_hex )),
                          *mission, view );
      break;
    case B_ID_BLD_DEL:
      mission->DeleteBuilding( map->GetBuilding( selected_hex ) );
      break;

    case B_ID_EVENTS:
      new EdEventsWindow( *mission, view );
      break;

    case B_ID_NMW_CREATE: {
      NewMissionWindow *nmw = static_cast<NewMissionWindow *>(win);
      rc = NewMission( nmw->GetMapSize(), nmw->GetTileSet(),
                       nmw->GetUnitSet() );
      break; }
    }

  }

  return rc;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::Quit
// DESCRIPTION: Ask user for confirmation and leave the editor.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::Quit( void ) const {
  DialogWindow *req = new DialogWindow( NULL, "Do you really want to quit?",
                                        "_Yes|_No", 1, 0, view );
  req->SetButtonID( 0, GUI_QUIT );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::ShowContextMenu
// DESCRIPTION: Open a pop-up menu with with context-dependent options.
// PARAMETERS : clicked - hex coordinates of the mouse click or -1,-1 if
//                        no specific hex was selected
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::ShowContextMenu( const Point &clicked ) {
  MenuWindow *menu = new MenuWindow( PROGRAMNAME, this, view );

  if ( mission ) {
    if ( clicked != Point(-1,-1) ) {
      Map *map = mv.GetMap();
      Unit *u = map->GetUnit( clicked );
      selected_hex = clicked;

      // tile sub-menu
      menu->AddMenu( 0, 0, "_Tile" );
      menu->AddItem( 1, B_ID_TILE_GRAB, 0, "_Grab" );
      menu->AddItem( 1, B_ID_TILE_SWAP,
                     (mode == ED_MODE_TERRAIN ? 0 : WIDGET_DISABLED),
                     "_Swap" );

      if ( u ) {  // unit sub-menu
        menu->AddMenu( 0, 0, "_Unit" );
        menu->AddItem( 1, B_ID_UNIT_INFO, WIDGET_DISABLED, "_Info" );  // FIXME
        menu->AddItem( 1, B_ID_UNIT_EDIT, 0, "_Edit..." );
        menu->AddItem( 1, B_ID_UNIT_DEL, 0, "_Delete" );
      }

      if ( map->IsBuilding( clicked ) ) {
        Building *b = map->GetBuilding( clicked );

        menu->AddMenu( 0, 0, "_Building" );
        if ( b ) {
          menu->AddItem( 1, B_ID_BLD_EDIT,  0, "_Edit..." );
          menu->AddItem( 1, B_ID_BLD_DEL,   0, "_Delete" );
        } else menu->AddItem( 1, B_ID_BLD_CREATE, 0, "_Create..." );
      }
    }

    menu->AddItem( 0, B_ID_EVENTS, 0, "_Events..." );
    menu->AddBar( 0 );
  }

  // file ops
  menu->AddMenu( 0, 0, "_Project" );
  menu->AddItem( 1, B_ID_NEW, 0, "_New..." );
  menu->AddItem( 1, B_ID_LOAD, 0, "_Load..." );
  menu->AddItem( 1, B_ID_SAVE, (mission ? 0 : WIDGET_DISABLED),
                 "_Save..." );
  menu->AddItem( 1, B_ID_EXPORT, (mission ? 0 : WIDGET_DISABLED),
                 "_Export..." );
  menu->AddItem( 1, B_ID_VALIDATE, (mission ? 0 : WIDGET_DISABLED),
                 "_Validate" );
  menu->AddItem( 1, B_ID_SETTINGS, (mission ? 0 : WIDGET_DISABLED),
                 "Se_ttings" );
  menu->AddItem( 1, B_ID_QUIT, 0, "_Quit..." );

  menu->Layout();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::SwapTiles
// DESCRIPTION: Replace each occurence of one tile on the map with
//              another tile.
// PARAMETERS : t1 - identifier of the tiles to be replaced
//              t2 - identifier of the tiles to replace t1
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::SwapTiles( unsigned short t1, unsigned short t2 ) {
  Map *map = mv.GetMap();
  Point p;

  for ( p.y = 0; p.y < map->Height(); ++p.y ) {
    for ( p.x = 0; p.x < map->Width(); ++p.x ) {
      if ( map->HexTypeID(p) == t1 ) map->SetHexType( p, t2 );
    }
  }

  mv.Draw();
  Show( mv );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdWindow::ValidateMission
// DESCRIPTION: Check mission integrity.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdWindow::ValidateMission( void ) const {
  string msg;

  if ( mission->Validate( msg ) > 0 ) {
    new MessageWindow( "Validation Results", msg.c_str(), view );
  } else {
     new NoteWindow( "Validation successful",
         "No errors or warnings have been detected.", 0, view );
  }
}

