/* mkunitset -- create a unit set for Crimson Fields
   Copyright (C) 2000-2004 Jens Granseuer

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/* create a data set for CF, containing graphics and definitions for units

   currently, all data (except the graphics) is hardcoded into
   the program

   level designers can specify custom data files to adjust the
   graphics and/or units to their likes
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "SDL.h"

#include "lang.h"
#include "ccompat.h"

#define NUM_SOUNDS		0

#define FID_SET			MakeID('U','S','E','T')

extern "C" {
extern const unsigned short NUM_UNIT_TYPES;
extern const struct UnitType UModels[];
extern const char *UNames_en[];
extern const char *UNames_de[];

const char *set_sounds[] = { NULL };

struct DatHeader {
  unsigned long fid;
  unsigned short units;
  unsigned char sounds;
};

extern int save_image_data( const char *filename, int trans, SDL_RWops *out );
}

int main( int argc, char *argv[] ) {
  int status, i;
  unsigned char num;
  struct DatHeader dhead = { FID_SET, NUM_UNIT_TYPES, NUM_SOUNDS };

  if ( argc <= 3 ) {
    fprintf(stderr, "Too few of arguments\n"
                    "Usage: %s <graphics.bmp> [<unit1.bmp>...] <outfile>\n",
                    argv[0] );
    exit(-1);
  }


  if ( SDL_Init(0) < 0 ) {
    fprintf(stderr, "Couldn't init SDL: %s\n", SDL_GetError());
    exit(-1);
  }
  atexit(SDL_Quit);

  File out( argv[argc-1] );
  if ( !out.Open( "wb" ) ) {
    fprintf(stderr, "Couldn't open output out.%s\n", argv[argc-1] );
    exit(-1);
  }

  status = 0;

  /* write data out.header */
  out.Write32( dhead.fid );
  out.Write16( dhead.units );

  /* write sound out.names */
  out.Write8( dhead.sounds );
  for ( i = 0; i < NUM_SOUNDS; ++i ) {
    unsigned char len = strlen( set_sounds[i] );
    out.Write8( len );
    out.WriteS( set_sounds[i] );
  }

  /* write unit data */
  Locale locale;
  Language lang_en;
  Language lang_de;
  lang_en.SetID( "en" );
  lang_de.SetID( "de" );

  for ( i = 0; i < NUM_UNIT_TYPES; ++i ) {
    out.Write16( UModels[i].ut_terrain );
    out.Write16( UModels[i].ut_image );
    out.Write16( UModels[i].ut_flags );
    out.Write( &UModels[i].ut_moves, 20 );
    lang_en.AddMsg( UNames_en[i] );
    lang_de.AddMsg( UNames_de[i] );
  }
  locale.AddLanguage( lang_en );
  locale.AddLanguage( lang_de );
  locale.Save( out );

  /* load graphics */
  status = save_image_data( argv[1], 1, out.GetFD() );

  /* load optional unit portraits */
  num = (unsigned char)(argc - 3);
  out.Write8( num );
  for ( i = 0; (i < num) && (status == 0); ++i ) {
    status = save_image_data( argv[2 + i], 0, out.GetFD() );
  }

  return status;
}

