
// Buffer pool.  Contains 8 buffers, each 2K (512 by 32).  Each buffer
// is a dual-ported RAM.  Port A on each of them is indirectly connected 
// to the wishbone bus by a bridge.  Port B may be connected any one of the
// 6 FIFO-like streaming interaces, or disconnected.  The wishbone bus
// provides access to all 8 buffers, and also controls the connections
// between the ports and the buffers, allocating them as needed.

// wb_adr is 16 bits -- 
//  bits 13:11 select which buffer
//  bits 10:2 select line in buffer
//  bits 1:0 are unused (32-bit access only)

module buffer_pool
  (input wb_clk_i,
   input wb_rst_i,
   input wb_we_i,
   input wb_stb_i,
   input [15:0] wb_adr_i,
   input [31:0] wb_dat_i,   
   output reg [31:0] wb_dat_o,
   output reg wb_ack_o,
   output wb_err_o,
   output wb_rty_o,
   
   input stream_clk,
   input stream_rst,

   input set_stb, input [7:0] set_addr, input [31:0] set_data,
   
   // Write Interfaces
   input [31:0] wr0_dat_i, input wr0_write_i, input wr0_done_i, output wr0_ready_o, output wr0_full_o,
   input [31:0] wr1_dat_i, input wr1_write_i, input wr1_done_i, output wr1_ready_o, output wr1_full_o,
   
   // Read Interfaces
   output [31:0] rd0_dat_o, input rd0_read_i, input rd0_done_i, output rd0_ready_o, output rd0_empty_o,
   output [31:0] rd1_dat_o, input rd1_read_i, input rd1_done_i, output rd1_ready_o, output rd1_empty_o
   );

   assign wb_err_o = 1'b0;  // Unused for now
   assign wb_rty_o = 1'b0;  // Unused for now

   wire sel_a [0:7];

   wire [2:0]  which_buf = wb_adr_i[13:11];   // address 15:14 selects the buffer pool
   wire [8:0]  buf_addra = wb_adr_i[10:2];     // ignore address 1:0, 32-bit access only
 
   decoder_3_8 dec(.sel(which_buf),.res(sel_a));

   genvar      i;

   wire [8:0]  fl[0:7];
   wire [8:0]  ll[0:7];
   wire [3:0]  step[0:7];
   wire        read_go[0:7];
   wire        write_go[0:7];
   wire [1:0]  read_port[0:7];
   wire [1:0]  write_port[0:7];
   wire [3:0]  dummy[0:7];
   wire        changed [0:7];

   wire [31:0] buf_doa[0:7];
   
   wire        buf_enb[0:7];
   wire        buf_web[0:7];
   wire [8:0]  buf_addrb[0:7];
   wire [31:0] buf_dib[0:7];
   wire [31:0] buf_dob[0:7];
   
   wire        done[0:7];

   wire [31:0] wr_dat_i[0:7];
   wire        wr_write_i[0:7];
   wire        wr_done_i[0:7];
   wire        wr_ready_o[0:7];
   wire        wr_full_o[0:7];

   wire [31:0] rd_dat_o[0:7];
   wire        rd_read_i[0:7];
   wire        rd_done_i[0:7];
   wire        rd_ready_o[0:7];
   wire        rd_empty_o[0:7];
   
   
   generate
      for(i=0;i<8;i=i+1)
	begin : gen_buffer
	   setting_reg #(.my_addr(i)) 
	     sreg(.clk(stream_clk),.rst(stream_rst),.strobe(set_stb),.addr(set_addr),.in(set_data),
		  .out({dummy[i],read_port[i],write_port[i],write_go[i],read_go[i],step[i],ll[i],fl[i]}),.changed(changed[i]));
	   buffer_2k buffer_2k
	     (.clka(wb_clk_i),.ena(wb_stb_i & sel_a[i]),.wea(wb_we_i),
	      .addra(buf_addra),.dia(wb_dat_i),.doa(buf_doa[i]),
	      .clkb(stream_clk),.enb(buf_enb[i]),.web(buf_web[i]),.addrb(buf_addrb[i]),.dib(buf_dib[i]),.dob(buf_dob[i]));
	   fifo_int fifo_int
	     (.clk(stream_clk),.rst(stream_rst),.firstline(fl[i]),.lastline(ll[i]),
	      .step(step[i]),.read_go(changed[i]&read_go[i]),.write_go(changed[i]&write_go[i]),.done(done[i]),
	      .en_o(buf_enb[i]),.we_o(buf_web[i]),.addr_o(buf_addrb[i]),.dat_to_buf(buf_dib[i]),.dat_from_buf(buf_dob[i]),
	      .wr_dat_i(wr_dat_i[i]),.wr_write_i(wr_write_i[i]),.wr_done_i(wr_done_i[i]),
	      .wr_ready_o(wr_ready_o[i]),.wr_full_o(wr_full_o[i]),
	      .rd_dat_o(rd_dat_o[i]),.rd_read_i(rd_read_i[i]),.rd_done_i(rd_done_i[i]),
	      .rd_ready_o(rd_ready_o[i]),.rd_empty_o(rd_empty_o[i])
	      );
	end
   endgenerate
   
   always @(posedge wb_clk_i)
     if(wb_stb_i)
       wb_dat_o <= buf_doa[which_buf];
   
   
   always @(posedge wb_clk_i)
     wb_ack_o <= wb_stb_i;
   
endmodule // buffer_pool

