

// FIFO Interface to the 2K buffer RAMs

module fifo_int
  (// Control Interface
   input clk,
   input rst,
   input [8:0] firstline,
   input [8:0] lastline,
   input [3:0] step,
   input read_go,
   input write_go,
   output done,
   
   // Buffer Interface
   output en_o,
   output we_o,
   output reg [8:0] addr_o,
   output [31:0] dat_to_buf,
   input [31:0] dat_from_buf,

   // Write FIFO Interface
   input [31:0] wr_dat_i,
   input wr_write_i,
   input wr_done_i,
   output wr_ready_o,
   output wr_full_o,
   
   // Read FIFO Interface
   output [31:0] rd_dat_o,
   input rd_read_i,
   input rd_done_i,
   output rd_ready_o,
   output rd_empty_o
   );
   
   // FIXME do we want to be able to interleave reads and writes?
   // FIXME do we want rd_ack or rd_req?
   
   wire      rd_en, wr_en;
   
   assign    dat_to_buf = wr_dat_i;
   assign    rd_dat_o = dat_from_buf;
   assign    en_o = rd_en | wr_en;

   localparam IDLE = 3'd0;
   localparam PRE_READ = 3'd1;
   localparam READING = 3'd2;
   localparam WRITING = 3'd3;

   reg [3:0]  state;
   
   always @(posedge clk)
     if(rst)
       begin
	  addr_o <= 0;
	  state <= IDLE;
       end
     else 
       case(state)
	 IDLE :
	   if(read_go)
	     begin
		addr_o <= firstline;
		state <= PRE_READ;
	     end
	   else if(write_go)
	     begin
		addr_o <= firstline;
		state <= WRITING;
	     end
	 
	 PRE_READ :
	   begin
	      state <= READING;
	      addr_o <= addr_o + 1;
	   end

	 READING :
	   if(rd_read_i)
	     begin
		addr_o <= addr_o + 1;
		if(addr_o == lastline + 9'd1)
		  state <= IDLE;
	     end

	 WRITING :
	   begin
	      if(wr_write_i)
		begin
		   addr_o <= addr_o + 1;
		   if(addr_o == lastline)
		     state <= IDLE;
		end
	      if(wr_done_i)
		state <= IDLE;
	   end

       endcase // case(state)

   // FIXME read side ignores rd_done_i for now
   
   assign rd_en = (state == PRE_READ) || ((state == READING) && rd_read_i);
   assign rd_empty_o = (state != READING) && (state != PRE_READ);
   assign rd_ready_o = (state == READING);

   assign wr_en = (state == WRITING) && wr_write_i;  // IF this is a timing problem, we could always enable when in this state
   assign we_o = (state == WRITING) && wr_write_i;  // IF this is a timing problem, we could always write when in this state
   assign wr_full_o = (state != WRITING);
   assign wr_ready_o = (state == WRITING);
   
endmodule // fifo_int
