#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <cstdio>
#include <cstring>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <netinet/ip.h>
#include <net/if.h>
#include <assert.h>
#include <pcap.h>
#include <errno.h>
#include "network.h"


char Network::errbuf[PCAP_ERRBUF_SIZE] = {};

/* get default network interface */
void Network::get_dev(char* dev)
{
        char* tmp;
        tmp = pcap_lookupdev(errbuf);
        if (tmp == NULL) {
                fprintf(stderr, "Pcap find no default device: %s\n", errbuf);
                exit(1);
        }
        memcpy(dev, tmp, strlen(tmp)+1);
	    printf("Public NIC: %s\n", dev);
}

/* get IP address on a specific network interface */
void Network::get_ip_dev(char* ipaddr, char* interface)
{
        int fd;
        struct ifreq ifr;
        fd = socket(AF_INET, SOCK_DGRAM, 0);
        ifr.ifr_addr.sa_family = AF_INET;
        strncpy(ifr.ifr_name, interface, IFNAMSIZ-1);
        ioctl(fd, SIOCGIFADDR, &ifr);
        close(fd);
        strncpy(ipaddr, inet_ntoa(((struct sockaddr_in *)&(ifr.ifr_addr))->sin_addr), 20);
	printf("Public IP: %s\n", ipaddr);
}





/* convert an IP string to unsigned integer in network order */
void Network::get_ip_port_from_sockaddr(struct sockaddr_in *addr, u_int32_t *ip, u_int16_t *port)
{
	*port = addr->sin_port;
	*ip = addr->sin_addr.s_addr;
}


/* convert IP of character string format to integer in network order */
u_int32_t Network::ip_atoi(const char* addr)
{
        struct in_addr inaddr;
        if(inet_aton(addr, &inaddr) == 0){
		fprintf(stderr, "ip_atoi() the address is invalid!\n");
		return -1;
	}
        return inaddr.s_addr;
}



/* make a sockaddr_in structure
 * ip_int and port_int are in network order
 * */
struct sockaddr_in * Network:: make_sockaddr(u_int32_t ip_int, u_int16_t port_int){
	struct sockaddr_in *addr;
	addr = (struct sockaddr_in *)malloc(sizeof(struct sockaddr_in));
	if(NULL == addr){
		perror("make_sockaddr: malloc failed\n");
		exit(1);
	}
	memset(addr, 0, sizeof(addr));
	addr->sin_family = AF_INET;
	addr->sin_addr.s_addr = ip_int;
	addr->sin_port = port_int;
	return addr;
}


int Network::setnonblock(int fd)
{
        int flags;
        if ((flags = fcntl(fd, F_GETFL)) == -1) {
		perror("fcntl F_GETFL failed!\n");
                return -1;
        }

        if (fcntl(fd, F_SETFL, flags | O_NONBLOCK) == -1) {
		perror("fcntl F_SETFL failed!\n");
                return -1;
        }
        return 0;
}

/* improved connect() function
 * return 0 if success
 * return -1 if connection refused, timeout, or network problem
 * otherwise, exit the program
 * */
int Network::Connect(int sockfd, const struct sockaddr *addr, socklen_t addrlen){
	int retval;
	retval = connect(sockfd, addr, addrlen);
	if(retval == -1){
		switch(errno){
			case ECONNREFUSED:
				fprintf(stderr, "connection to server refused!\n");
				break;
			case ETIMEDOUT:
				fprintf(stderr, "connection to server timed out!\n");
				break;
			case ENETUNREACH:
				fprintf(stderr, "network unreachable!\n");
				break;
			default:
				perror("connection to server error!\n");
				exit(1);
		}
	}
	return retval;

}

/* improved send() function
 * return -1 if EPIPE or EAGAIN
 * or exit if other errors
 * or return number of bytes
 * */
ssize_t Network::Send(int sockfd, const void *buf, size_t len)
{
	ssize_t retval;
	retval = send(sockfd, buf, len, MSG_NOSIGNAL);
	if(retval == -1){
		switch(errno){
			case EPIPE:
				printf("The server agent closes the connection!\n");
				break;
			case EAGAIN:
			/* for non-blocking socket */
				printf("send buffer to server is full\n!");
				break;
			default:
				perror("send() to server error!\n");
				exit(1);
		}
	}
	return retval;
}
