#include <PenguinPlay/SampleAF.h>

#ifndef PP_HAVE_AUDIOFILE_H
// this is here so somebody can compile it if they don't have audiofile
// (there obviously won't be any Audiofile support.)
PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

SampleAF::SampleAF()
{
     ppThrow (EUnsupportedEnvironment,
              "Audiofile support was not compiled in.");
}

SampleAF::~SampleAF()
{
}

bool SampleAF::Load(const char*)                             { return false; }
bool SampleAF::Save(const char*, const Sample& sample) const { return false; }

PP_I_NAMESPACE_END
PP_NAMESPACE_END

#else /* PP_HAVE_AUDIOFILE_H */

#include <string>
#include <iostream>

#include <sys/types.h> // u_int8_t, u_int16_t

extern "C"
{
#include <audiofile.h>
}

#include <dlfcn.h> // dlopen, dlsym, dlerror, dlclose

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

/*
   A class that dynamically loads useful functions from libaudiofile.so.
   This is done so that there is no requirement for the user to have audiofile.
   If the user would like audiofile support they need to download the library.

   Note: it is possible to get audiofile to compile under windows but the official
   release doesn't support it. Maybe I should get my act together and send the
   maintainer some patches.
*/
class AFprivate
{
public:
  AFprivate();
  ~AFprivate();

  bool LoadFunctions();

  typedef AFfilehandle (*afOpenFile_t)(const char*, const char*, AFfilesetup);
  typedef int          (*afCloseFile_t)(AFfilehandle);
  typedef AFframecount (*afGetFrameCount_t)(AFfilehandle, int);
  typedef int          (*afGetChannels_t)(AFfilehandle, int);
  typedef double       (*afGetRate_t)(AFfilehandle, int);
  typedef double       (*afGetSampleFormat_t)(AFfilehandle, int, int*, int*);
  typedef int          (*afSetVirtualByteOrder_t)(AFfilehandle, int, int);
  typedef int          (*afReadFrames_t)(AFfilehandle, int, void*, int);

  afOpenFile_t            m_afOpenFile;
  afCloseFile_t           m_afCloseFile;
  afGetFrameCount_t       m_afGetFrameCount;
  afGetChannels_t         m_afGetChannels;
  afGetRate_t             m_afGetRate;
  afGetSampleFormat_t     m_afGetSampleFormat;
  afSetVirtualByteOrder_t m_afSetVirtualByteOrder;
  afReadFrames_t          m_afReadFrames;
  void*                   m_handle;
};

AFprivate::AFprivate()
  :  m_afOpenFile(0),
     m_afCloseFile(0),
     m_afGetFrameCount(0),
     m_afGetChannels(0),
     m_afGetRate(0),
     m_afGetSampleFormat(0),
     m_afSetVirtualByteOrder(0),
     m_afReadFrames(0)
{
  m_handle = dlopen("libaudiofile.so", RTLD_LAZY);
  if (m_handle == 0)
    ppThrow (EAccessFailure, "Can't open libaudiofile.so");

  if (!LoadFunctions())
    ppThrow (EUnsupportedEnvironment, "Can't load needed symbols from libaudiofile.so");
}

AFprivate::~AFprivate()
{
  try
    {
      if (m_handle != 0)
        {
          dlclose(m_handle);
          m_handle = 0;
        }
    }
  catch (...)
    {
    }
}

bool AFprivate::LoadFunctions()
{
  m_afOpenFile            = (afOpenFile_t)dlsym(m_handle, "afOpenFile");
  m_afCloseFile           = (afCloseFile_t)dlsym(m_handle, "afCloseFile");
  m_afGetFrameCount       =
    (afGetFrameCount_t)dlsym(m_handle, "afGetFrameCount");
  m_afGetChannels         = (afGetChannels_t)dlsym(m_handle, "afGetChannels");
  m_afGetRate             = (afGetRate_t)dlsym(m_handle, "afGetRate");
  m_afGetSampleFormat     =
    (afGetSampleFormat_t)dlsym(m_handle, "afGetSampleFormat");
  m_afSetVirtualByteOrder =
    (afSetVirtualByteOrder_t)dlsym(m_handle, "afSetVirtualByteOrder");
  m_afReadFrames          = (afReadFrames_t)dlsym(m_handle, "afReadFrames");

  // make sure we loaded all the symbols
  return   ((m_afOpenFile            != 0) &&
            (m_afCloseFile           != 0) &&
            (m_afGetFrameCount       != 0) &&
            (m_afGetChannels         != 0) &&
            (m_afGetRate             != 0) &&
            (m_afGetSampleFormat     != 0) &&
            (m_afSetVirtualByteOrder != 0) &&
            (m_afReadFrames          != 0));
}

SampleAF::SampleAF()
{
}

SampleAF::~SampleAF()
{
}


static void Copy16Interleaved(u_int16_t*    dest,
                              u_int16_t*    src,
                              unsigned long length)
{
  while (length--) {*dest++ = *src++; ++src;}
}

static void Copy8Interleaved(u_int8_t*     dest,
                             u_int8_t*      src,
                             unsigned long  length)
{
  while (length--) {*dest++ = *src++; ++src; }
}


bool SampleAF::Load(const char* filename)
{
  AFfilehandle fh;
  int in_format;
  int in_width;
  int in_channels;
  int frame_count;
  int bytes_per_frame;

  AFprivate af;

  fh = af.m_afOpenFile(filename, "r", 0);
  if (fh==0)
    return false;

  frame_count = af.m_afGetFrameCount(fh, AF_DEFAULT_TRACK);
  in_channels = af.m_afGetChannels(fh, AF_DEFAULT_TRACK);
  m_frequency = (int)af.m_afGetRate(fh, AF_DEFAULT_TRACK);
  af.m_afGetSampleFormat(fh, AF_DEFAULT_TRACK, &in_format, &in_width);
  af.m_afSetVirtualByteOrder(fh, AF_DEFAULT_TRACK, AF_BYTEORDER_LITTLEENDIAN);

  ppDebug1 ("frames     %d", frame_count);
  ppDebug1 ("channels   %d", in_channels);
  ppDebug1 ("frequency  %d", m_frequency);
  ppDebug1 ("bits       %d", in_width);

  if (in_width == 8)
    Set8Bit();
  else if (in_width == 16)
    Set16Bit();
  else
    return false;


  bytes_per_frame = (in_width * in_channels) / 8;

  m_length = bytes_per_frame * frame_count;

  if (in_channels == 1)
    {
      SetMono();
      m_data0 = new u_int8_t [m_length];
      // get sample data
      af.m_afReadFrames(fh, AF_DEFAULT_TRACK, m_data0, frame_count);
    }
  else if (in_channels == 2)
    {
      SetStereo();
      unsigned char* buffer = new u_int8_t [frame_count*2*in_width/8];
      af.m_afReadFrames(fh, AF_DEFAULT_TRACK, buffer, frame_count);

      m_data0 = new u_int8_t [m_length];
      m_data1 = new u_int8_t [m_length];
      if (Is8Bit())
        {
          Copy8Interleaved(m_data0, buffer, frame_count);
          Copy8Interleaved(m_data1, buffer+1, frame_count);
        }
      else
        {
          Copy16Interleaved((u_int16_t*)m_data0,
                            (u_int16_t*)buffer,
                            frame_count);
          Copy16Interleaved((u_int16_t*)m_data1,
                            (u_int16_t*)buffer+1,
                            frame_count);
        }
      delete buffer;
    }
  else
    {
      return false;
    }

  // close the file
  if (af.m_afCloseFile(fh) != 0)
    return false;

  if (m_frequency == 0)
    m_frequency = 44100;

  return true;
}

bool SampleAF::Save(const char* filename, const Sample& sample) const
{
  return false;
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END

#endif // PP_HAVE_AUDIOFILE_H
