//=============================================================================
// $Id: Sequencer.cpp,v 1.2 1999/11/28 23:47:26 creinig Exp $
//-----------------------------------------------------------------------------
// $Log: Sequencer.cpp,v $
// Revision 1.2  1999/11/28 23:47:26  creinig
// adapted sources to use ppconfig(_win32).h
//
// Revision 1.1.1.1  1999/11/06 11:54:01  creinig
// Back in CVS at last
//
//=============================================================================
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>

#include <PenguinPlay/PenguinPlay.h>

#include <PenguinPlay/Sample.h>
#include <PenguinPlay/Sequencer.h>
#include <PenguinPlay/Channel.h>

// config.h gets included through PenguinPlay.h
#if defined(PP_HAVE_SYS_SOUNDCARD_H)
#include <sys/soundcard.h>
#elif defined(PP_HAVE_MACHINE_SOUNDCARD_H)
#include <machine/soundcard.h>
#endif

#if defined(PP_HAVE_AWE_VOICE_H)
#include <awe_voice.h>
#define PP_HAVE_AWE 1
#elif defined(PP_HAVE_SYS_AWE_VOICE_H)
#include <sys/awe_voice.h>
#define PP_HAVE_AWE 1
#elif defined(PP_HAVE_LINUX_AWE_VOICE_H)
#include <linux/awe_voice.h>
#define PP_HAVE_AWE 1
#endif

//#include <linux/ultrasound.h>

#include <math.h>

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN


//=============================================================================
// compulsory soundcard.h code
//=============================================================================
SEQ_DEFINEBUF(2048);

static int seqfd;

void seqbuf_dump()
{
  if (_seqbufptr)
    if (write(seqfd, _seqbuf, _seqbufptr) == -1)
      {
	perror("write /dev/sequencer");
	exit(-1);
      }
  _seqbufptr = 0;
}
//----------------------------------------------------------------------------
// static variables
const char* const Sequencer::DEV_SEQUENCER = "/dev/sequencer";

//----------------------------------------------------------------------------

// public

///  Sequencer Constructor.
///   Query the sequencer for the synth devices. If there is a wavetable
///   synth then use it.
Sequencer::Sequencer()
  : m_timeout(0),
    m_voices_allocated(1),
    m_channels(1)
{
  synth_info si;
  int        nrsynths;

  if ((seqfd = open(DEV_SEQUENCER, O_RDWR | O_NDELAY)) < 0)
    {
      perror(DEV_SEQUENCER);
      ppThrow (EAccessFailure, "Can't open device");
    }
  if (ioctl(seqfd, SNDCTL_SEQ_NRSYNTHS, &nrsynths) == -1)
    {
      perror("there is no soundcard");
      ppThrow (EHardwareUnavailable, "No Sample Synth Found");
    }

  ppDebug1 ("%d nrsynths", nrsynths);

  m_device = -1;

  for (int i = 0; i < nrsynths; ++i)
    {
      si.device = i;
      if (ioctl(seqfd, SNDCTL_SYNTH_INFO, &si) == -1)
	{
	  ppThrow (EHardwareUnavailable, "cannot get info on soundcard");
	}
#ifdef PP_DEBUG
      ppDebug1 ("-----------------------------");
      ppDebug1 (" device  : %d" , i);
      ppDebug1 (" name    : \"%s\"", si.name);
      switch (si.synth_type)
        {
        case SYNTH_TYPE_FM:
          ppDebug1 (" type    : SYNTH_TYPE_FM");
          break;

        case SYNTH_TYPE_SAMPLE:
          ppDebug1 (" type    : SYNTH_TYPE_SAMPLE");
          break;

        case SYNTH_TYPE_MIDI:
          ppDebug1 (" type    : SYNTH_TYPE_MIDI");
          break;

        default:
          ppDebug1 (" type    : unknown");
          break;
        }

      switch(si.synth_subtype)
        {
        case FM_TYPE_ADLIB:
          ppDebug1 (" subtype : FM_TYPE_ADLIB");
          break;

        case FM_TYPE_OPL3:
          ppDebug1 (" subtype : FM_TYPE_OPL3");
          break;

        case SAMPLE_TYPE_GUS:
          ppDebug1 (" subtype : SAMPLE_TYPE_GUS");
          break;

        case SAMPLE_TYPE_AWE32:
          m_is_awe = true;
#ifdef PP_HAVE_AWE
          AWE_DEBUG_MODE(m_device,1);
#endif
          ppDebug1 (" subtype : SAMPLE_TYPE_AWE32");
          break;

        case MIDI_TYPE_MPU401:
          ppDebug1 (" subtype : MIDI_TYPE_MPU401");
          break;

        default:
          ppDebug1 (" subtype : unknown");
          break;
        }

#endif
      if (si.synth_type == SYNTH_TYPE_SAMPLE)
	{
          //          Init(i)
          // -- don't call Init it does strange things to GetMemavl
          int memavl;

          memavl = GetMemavl(i);

          ppDebug1 (" memory available: %d", memavl);
          Reset(i);
          memavl = GetMemavl(i);

          ppDebug1 (" memory available: %d", memavl);
          ppDebug1 ("-----------------------------");

#if 0
          if (!memavl)
            continue;
#endif
	  m_device = i;


          StartTimer();

          SEQ_VOLUME_MODE(m_device, VOL_METHOD_LINEAR);
          seqbuf_dump();

          break;
	}
    }
}

Sequencer::~Sequencer()
{
  try
    {
      StopTimer();
      Sync();
      seqbuf_dump();
      close(seqfd);
    }
  catch (...)
    {
    }
}

Audio::Result Sequencer::Play(Sample& sample)
{
  int v = AllocateVoice(sample);
  Channel* c = AllocateChannel();
  c->SetVoice(v);
  c->Start();
  Wait(100*sample.GetSamples()/sample.GetFrequency());
  c->Stop();
  Sync();
  DeleteChannel(c);

  // wait till sample stops playing?
  // deallocate channel
  // deallocate voice

  return SUCCESS;
}

int Sequencer::GetMemavl(int device)
{
  int temp = (device == -1) ? m_device : device;

  ioctl(seqfd, SNDCTL_SYNTH_MEMAVL, &temp);

  return temp;
}

void Sequencer::Init(int device)
{
  // this does strange things to the memory available function on an AWE 64
  // -- dont use it -- theres something going wrong
  if (device == -1) device = m_device;
#ifdef PP_HAVE_AWE
  AWE_INITIALIZE_CHIP(seqfd,device);
#endif
}

// private



int Sequencer::AllocateVoice(const Sample& sample, int voice)
{
  if (voice == -1)
    voice = VoicesAllocated();

  patch_info* pi = 0;

  pi=(patch_info*)new unsigned char [sizeof(patch_info)+sample.GetLength()];

  pi->key       = GUS_PATCH;
  pi->device_no = m_device;
  pi->instr_no  = voice;

  pi->mode = 0;

  ppDebug1 (" voice %d", voice);

  if (sample.Is16Bit())
    {
      ppDebug1 ("16 bit ");
      pi->mode |= WAVE_16_BITS;
    }
  if (!sample.IsSigned())
    {
      ppDebug1 ("unsigned ");
      pi->mode |= WAVE_UNSIGNED;
    }
  if (sample.IsLooped())
    {
      ppDebug1 ("looping ");
      pi->mode |= WAVE_LOOPING;
    }
  if (sample.IsBIDI())
    {
      ppDebug1 ("bidi ");
      pi->mode |= WAVE_BIDIR_LOOP;
    }


  ppDebug1 ("mode %x", pi->mode);

  // can also set vibrato, tremolo, sustain and envelopes

  pi->len        = sample.GetLength();

  pi->loop_start = sample.GetLoopStart();
  pi->loop_end   = sample.GetLoopEnd();



  if (sample.Is16Bit())
    {
      // patch_info uses length in bytes, Sample uses loops in samples -convert
      pi->loop_start <<= 1;
      pi->loop_end   <<= 1;
    }


  // note freq * 1000 eg middle A is 440*1000
  pi->base_freq = sample.GetFrequency();
  //  pi->base_note = 130812 ; // 440 / ( 2 ^ (21 / 12)) * 1000 (taken from xmp)
  pi->base_note = 261625;

  pi->high_note = 0x7fffffff;
  pi->low_note  = 0;

  ppDebug1 (" base_freq %d", pi->base_freq);
  ppDebug1 (" base_note %d", pi->base_note);

  pi->panning  = sample.GetX(); //-128= left , 127 = right;
  pi->detuning = 0;

  // env_rate and env_offset ?
  // tremolo and vibrato ?
  // scale ?
  pi->volume = sample.GetVolume();

  memcpy(pi->data, sample.GetData(), sample.GetLength());
  SEQ_WRPATCH(pi, sizeof(*pi) + pi->len - 1);
  ++m_voices_allocated;
  seqbuf_dump();
  return voice;
}


int Sequencer::VoicesAllocated() const
{
  return m_voices_allocated;
}


Sequencer::Result Sequencer::SetLocation(int x,
                                               int y,
                                               int z,
                                               int channel)
{
  Result result;
  result = SetX(x, channel);
  if (result != SUCCESS)
    return result;
  result = SetY(y, channel);
  if (result != SUCCESS)
    return result;
  result = SetZ(z, channel);
  return result;
}

Sequencer::Result Sequencer::SetX(int x, int channel)
{
  ppDebug1 (" set panning, device %d, channel %d, value %d",
            m_device,
            channel,
	    x);

  SEQ_CONTROL(m_device, channel, CTL_PAN, x);
  return SUCCESS;
}


Sequencer::Result Sequencer::SetY(int y, int channel)
{
  return NOT_AVAILABLE;
}

Sequencer::Result Sequencer::SetZ(int z, int channel)
{
  return NOT_AVAILABLE;
}

int Sequencer::GetX(int channel)
{
  return -1;
}

int Sequencer::GetY(int channel)
{
  return -1;
}

int Sequencer::GetZ(int channel)
{
  return -1;
}

/// Set the expression volume.
/// The final volume will be "main volume * expression * velocity".
Sequencer::Result Sequencer::SetVolume(Volume volume, int channel)
{
  ppDebug1 (" set volume, device %d, channel %d, value %d",
            m_device,
            channel,
            volume);

  SEQ_CONTROL(m_device, channel, CTL_EXPRESSION, volume);
  return SUCCESS;
}


Sequencer::Volume Sequencer::GetVolume(int channel)
{
  return -1;
}

Sequencer::Result Sequencer::SetVelocity(Velocity, int channel)
{
  return NOT_AVAILABLE;
}

Sequencer::Velocity Sequencer::GetVelocity(int channel)
{
  return -1;
}


/// PROBLEM:
/// the sequencer expects me set the pitch of the voice by giving it a note
/// and pitch bend rather than a frequency.
/// If the voice is already playing it expects me
/// to bend the pitch of the note. Thus we need to convert the frequency
/// to a note and pitch bend . If the voice is already playing then we need
/// to  convert the frequency to a note, get the current note and pitch
/// bend and then calculate the difference.
///
/// SEQ_BENDER has a value from 0 to 16384,
/// and the center (no pitch shift) is 8192
Sequencer::Result Sequencer::SetFrequency(Frequency frequency,
                                                int       channel)
{
  if (frequency > 16384)
    return FAILURE;
  SEQ_BENDER(m_device, channel, frequency);
  return SUCCESS;
}

Sequencer::Frequency Sequencer::GetFrequency(int channel)
{
  return -1;
}


Sequencer::Result Sequencer::SetPosition(unsigned long position,
                                               int           channel)
{
  return NOT_AVAILABLE;
}

unsigned long Sequencer::GetPosition(int channel)
{

}



Sequencer::Result Sequencer::SetVoice(int voice, int channel)
{
  ppDebug1 (" set patch, device %d, voice %d, channel %d",
            m_device,
            voice,
            channel);

  SEQ_SET_PATCH(m_device, channel, voice);
  return SUCCESS;
}

int Sequencer::GetVoice(int channel)
{

}

void Sequencer::Reset(int device)
{
  ppDebug1 (" resetting samples");

  int temp = (device == -1) ? m_device : device;
  ioctl(seqfd, SNDCTL_SEQ_RESETSAMPLES, &temp);
}

void Sequencer::Wait(int x)
{
  SEQ_WAIT_TIME(x);
  seqbuf_dump();
}

void Sequencer::Sync()
{
  ioctl(seqfd, SNDCTL_SEQ_SYNC);
}


//---------
// chorus
//---------

void Sequencer::SetChorusMode(int mode)
{
#ifdef PP_HAVE_AWE
  if (m_is_awe)
    AWE_CHORUS_MODE(m_device, mode);
#endif
}

void Sequencer::SetChorusDepth(int channel, int depth)
{
  SEQ_CONTROL(m_device, channel, CTL_CHORUS_DEPTH, depth);
}

//--------
// reverb -- awe only
//--------

void Sequencer::SetReverbMode(int mode)
{
#ifdef PP_HAVE_AWE
  if (m_is_awe)
    AWE_REVERB_MODE(m_device, mode);
#endif
}

void Sequencer::SetReverbDepth(int channel, int depth)
{
  SEQ_CONTROL(m_device, channel, CTL_EXT_EFF_DEPTH, depth);
}



// notes

void Sequencer::StartNote(int channel, int note, int velocity)
{
  ppDebug1 (" start note, device %d, channel %d, note %d, velocity %d",
	    m_device,
	    channel,
	    note,
	    velocity);

  //  SEQ_CONTROL(m_device, channel, note, velocity);
  SEQ_START_NOTE(m_device, channel, note, velocity);
  //GUS_CHANNELON(m_device, channel, 0);
  seqbuf_dump();
}

void Sequencer::StopNote(int channel, int note, int velocity)
{
#ifdef PP_DEBUG
  ppDebug1 (" stop note, device %d, channel %d, note %d, velocity %d",
	    m_device,
	    channel,
	    note,
	    velocity);
#endif
  SEQ_STOP_NOTE(m_device, channel, note, velocity);
  seqbuf_dump();
}

void Sequencer::TerminateChannel(int channel)
{
#ifdef PP_HAVE_AWE
  AWE_TERMINATE_CHANNEL(m_device, channel);
#endif
}

Channel* Sequencer::AllocateChannel()
{
  return  new Channel(*this, m_channels++);
}

void Sequencer::DeleteChannel(Channel*& channel)
{
  delete channel;
  channel = 0;
  --m_channels;
}

int Sequencer::NextChannel()
{
  return m_channels;
}

/// This is used to set the synth device to use.
///
/// On my AWE 64 Value, when I cat /dev/sndstat I get
///
/// Synth devices:
/// 0: Yamaha OPL-3
/// 1: AWE32-0.4.2c (RAM512k)
bool Sequencer::SetDevice(int device)
{
  if (device < m_num_devices)
    m_device = device;
  return true;
}

long Sequencer::GetPlaybackRate()
{
  return 44100; // ?
}

// timers

void Sequencer::StartTimer ()
{
    SEQ_START_TIMER();
    seqbuf_dump();
}


void Sequencer::StopTimer ()
{
    SEQ_STOP_TIMER();
    seqbuf_dump();
}

void Sequencer::SetTimeout(long milliseconds)
{
  m_timeout = milliseconds;
}

void Sequencer::StartTimeout()
{
  //  StartTimer();
}

void Sequencer::StopTimeout()
{
  //  StopTimer();
}

void Sequencer::WaitForTimeout()
{
  Wait(m_timeout);
  Sync();
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END
