#! /usr/bin/env python
'''
An example of using the collision_resolver module

Pops up a window populated by randomly placed little squares.
You control the big square with the direction keys.
'''

from random import randint
import pygame
from pygame.locals import *
from collision_resolver import resolve_collisions


RESOLUTION = (600,400)
SPEED_X = 10
SPEED_Y = 10
ORIG_BACKGROUND = None

def vectorSum( v1, v2 ):
    return [ v1[0]+v2[0], v1[1]+v2[1] ]

def main():
    global ORIG_BACKGROUND, SPEED_X, SPEED_Y
    pygame.init()
    screen = pygame.display.set_mode(RESOLUTION)
    pygame.display.set_caption( 'Collision Resolver Example' )

    clock = pygame.time.Clock()

    avatar, avatarGroup, background = Start( screen )
    ORIG_BACKGROUND = background.copy()

    while True:
        clock.tick(2)

        for ev in pygame.event.get():
            if ev.type == QUIT:
                return #if there's a quit event, return
            if ev.type == KEYDOWN and ev.key == K_ESCAPE:
                return
            if ev.type == KEYDOWN and ev.key == K_LEFT:
                avatar.moveState[0] = -SPEED_X
            if ev.type == KEYDOWN and ev.key == K_RIGHT:
                avatar.moveState[0] = SPEED_X
            if ev.type == KEYDOWN and ev.key == K_DOWN:
                avatar.moveState[1] = SPEED_Y
            if ev.type == KEYDOWN and ev.key == K_UP:
                avatar.moveState[1] = -SPEED_Y
            if ev.type == KEYUP and ev.key in [K_LEFT,K_RIGHT]:
                avatar.moveState[0] = 0
            if ev.type == KEYUP and ev.key in [K_UP, K_DOWN]:
                avatar.moveState[1] = 0
            if ev.type == KEYDOWN and ev.key == K_x:
                SPEED_X += 5
            if ev.type == KEYDOWN and ev.key == K_y:
                SPEED_Y += 5
            if ev.type == KEYDOWN and ev.key == K_w:
                avatar.change_width(-5)
            if ev.type == KEYDOWN and ev.key == K_h:
                avatar.change_height(-5)

        #clear
        avatarGroup.clear( screen, background )

        #update
        avatarGroup.update()

        #draw
        avatarGroup.draw(screen)
        pygame.display.update()

def Start( screen ):
    background = pygame.Surface( RESOLUTION )
    off_black = (40,10,0)
    background.fill( off_black )

    # avatar will be a green square in the center of the screen
    avatar = Avatar()

    fixedBackgroundSprites = pygame.sprite.Group()
    for block in GenerateRandomBlocks(30, RESOLUTION):
        if not block.rect.colliderect(avatar.rect):
            fixedBackgroundSprites.add( block )
    fixedBackgroundSprites.draw( background )

    avatar.collidables = fixedBackgroundSprites

    avatarGroup = pygame.sprite.Group()
    avatarGroup.add( avatar )

    screen.blit( background, (0,0) )
    pygame.display.flip()

    return avatar, avatarGroup, background

def DrawVisit(sprite, limitsX, limitsY):
    screen = pygame.display.get_surface()
    if limitsX:
        pygame.draw.rect(screen, (200,0,0), sprite.rect, 3)
        pygame.display.flip()
        print 'visited', sprite, 'limitsX', limitsX
    if limitsY:
        pygame.draw.rect(screen, (255,0,0), sprite.rect, 3)
        pygame.display.flip()
        print 'visited', sprite, 'limitsY', limitsY

class SimpleSprite(pygame.sprite.Sprite):
    def __init__(self, surface):
        pygame.sprite.Sprite.__init__(self)
        self.image = surface
        self.rect = self.image.get_rect()

green = (5,255,5)
class Avatar(pygame.sprite.Sprite):
    def __init__(self):
        pygame.sprite.Sprite.__init__(self)
        self.image = pygame.Surface( (100,100) )
        self.image.fill( green )
        self.rect = self.image.get_rect()
        self.rect.center = (RESOLUTION[0]/2, RESOLUTION[1]/2)
        self.moveState = [0,0]
        self.collidables = None

    def change_width(self, change):
        self.rect = self.rect.inflate(change,0)
        self.image = pygame.Surface( self.rect.size )
        self.image.fill( green )

    def change_height(self, change):
        self.rect = self.rect.inflate(0,change)
        self.image = pygame.Surface( self.rect.size )
        self.image.fill( green )

    def draw_corner_line(self, corner):
        screen = pygame.display.get_surface()
        pygame.draw.line(screen, green, corner, 
                         vectorSum(corner, self.moveState)
                        )
    def update(self):
        if self.moveState[0] or self.moveState[1]:
            screen = pygame.display.get_surface()
            screen.blit( ORIG_BACKGROUND, (0,0) )
            self.draw_corner_line(self.rect.topleft)
            self.draw_corner_line(self.rect.topright)
            self.draw_corner_line(self.rect.bottomleft)
            self.draw_corner_line(self.rect.bottomright)
            pygame.display.flip()
        result = resolve_collisions(self.rect, 
                                    self.moveState[0], self.moveState[1],
                                    self.collidables, visitFn=DrawVisit)
        movedRect, xLimiter, yLimiter = result
        self.rect = movedRect
        
def GenerateRandomBlocks( howMany, positionBounds ):
    lowerColorBound = (100,100,100)
    upperColorBound = (200,200,200)

    lowerXBound, lowerYBound = 0,0
    upperXBound, upperYBound = positionBounds

    lowerWidthBound, lowerHeightBound = 30,30
    upperWidthBound, upperHeightBound = 60,60

    for i in range(howMany):
        color = [ randint(lowerColorBound[i],upperColorBound[i]) 
                  for i in range(3) ]
        pos = [ randint(lowerXBound, upperXBound), 
                randint(lowerYBound,upperYBound) ]
        size = [ randint(lowerWidthBound, upperWidthBound),
                 randint(lowerHeightBound, upperHeightBound) ]

        s = SimpleSprite( pygame.Surface(size) )
        s.image.fill( color )
        s.rect.topleft = pos
        yield s
    
def pathological_test():
    def MakeBadBlocks( howMany, positionBounds ):
        retval = [
                  SimpleSprite( pygame.Surface( (50,50) ) ), 
                  SimpleSprite( pygame.Surface( (50,50) ) ),
                 ]
        retval[0].rect.topleft = (240,250)
        retval[1].rect.topleft = (150,148)
        retval[0].image.fill( (200,200,100) )
        retval[1].image.fill( (100,100,100) )
        return retval

    global SPEED_X, SPEED_Y
    SPEED_X = 60
    SPEED_Y = 60
    global GenerateRandomBlocks
    GenerateRandomBlocks = MakeBadBlocks


if __name__ == '__main__':
    import sys
    if len(sys.argv) >= 2:
        if sys.argv[1] == 'pathological':
            pathological_test()
    print "-"*60
    print "Move the green block with the direction keys"
    print "Press x: change the green block's x speed"
    print "Press y: change the green block's y speed"
    print "Press w: change the green block's width"
    print "Press h: change the green block's height"
    print "-"*60
    main()
