#! /usr/bin/env python
'''
An example of using the scroll_group and scroll_bgmanager modules.

Pops up a window populated by randomly placed little squares.
You control the big square with the direction keys.

As the big square moves around the central part of the play area, the 
background will scroll to keep him on the screen.  If he moves around 
the edges of the play area, the background will not scroll, as there's 
nothing defined beyond that edge.  

If he stays near the center of the screen and only makes small movements, 
there will be no scrolling either, this is a feature to conserve 
processor time and can be turned off if desired.
'''
__author__ = 'sjbrown'

from random import randint
import pygame
from pygame.locals import *
import scroll_bgmanager
import scroll_group


RESOLUTION = (800,600)

def main():
    pygame.init()
    screen = pygame.display.set_mode(RESOLUTION)
    pygame.display.set_caption( 'Scrolling Background Example' )

    clock = pygame.time.Clock()

    avatar, avatarGroup, bgManager = Start( screen )

    while True:
        clock.tick(60)

        if [ev for ev in pygame.event.get() if ev.type == QUIT]:
            return #if there's a quit event, return

        downKeys = pygame.key.get_pressed()

        if downKeys[K_ESCAPE]:
            return
        if downKeys[K_LEFT]:
            avatar.rect.x -= 3
        if downKeys[K_RIGHT]:
            avatar.rect.x += 3
        if downKeys[K_DOWN]:
            avatar.rect.y += 3
        if downKeys[K_UP]:
            avatar.rect.y -= 3

        # Tell the bgManager where the Player's sprite is
        bgManager.NotifyPlayerSpritePos( avatar.rect )

        #clear
        avatarGroup.clear( screen )

        #update
        avatarGroup.update()

        #draw
        changedRects = avatarGroup.draw(screen)
        pygame.display.update( changedRects )

def Start( screen ):
    # Note that this is larger than the screen resolution
    bgWidth, bgHeight = 1000,1000

    background = pygame.Surface( (bgWidth, bgHeight) )
    off_black = (40,10,0)
    background.fill( off_black )

    fixedBackgroundSprites = pygame.sprite.Group()
    for block in GenerateRandomBlocks(20, (bgWidth, bgHeight)):
        fixedBackgroundSprites.add( block )
    fixedBackgroundSprites.draw( background )

    bgManager = scroll_bgmanager.BackgroundManager( screen, background )

    # avatar will be a white square in the center of the screen
    avatar = SimpleSprite( pygame.Surface( (100,100) ) )
    avatar.image.fill( (255,255,255) )
    avatar.rect.center = (RESOLUTION[0]/2, RESOLUTION[1]/2)

    avatarGroup = scroll_group.ScrollSpriteGroup( bgManager )
    avatarGroup.add( avatar )

    bgManager.BlitSelf( screen )
    pygame.display.flip()

    return avatar, avatarGroup, bgManager


class SimpleSprite(pygame.sprite.Sprite):
    def __init__(self, surface):
        pygame.sprite.Sprite.__init__(self)
        self.image = surface
        self.rect = self.image.get_rect()
        #self.displayRect = self.rect.move(0,0) #copy
    
def GenerateRandomBlocks( howMany, positionBounds ):
    lowerColorBound = (100,100,100)
    upperColorBound = (200,200,200)

    lowerXBound, lowerYBound = 0,0
    upperXBound, upperYBound = positionBounds

    for i in range(howMany):
        color = [ randint(lowerColorBound[i],upperColorBound[i]) 
                  for i in range(3) ]
        pos = [ randint(lowerXBound, upperXBound), 
                randint(lowerYBound,upperYBound) ]
        s = SimpleSprite( pygame.Surface( (50,50) ) )
        s.image.fill( color )
        s.rect.topleft = pos
        yield s
    


if __name__ == '__main__':
    print "Move the white block with the direction keys"
    print "You will see the scrolling effect"
    main()
