"""Example of efficient display of a rotating GIF image"""
from math import sin, cos, radians
import gc

# Start pygame
import pygame
from pygame.locals import QUIT, KEYDOWN, K_ESCAPE
pygame.init()

screen_width, screen_height = screen_size = (600, 400)
screen = pygame.display.set_mode(screen_size)

blue = (0, 0, 255, 255)
bg = pygame.Surface(screen_size, 0, screen)
bg.fill(blue)
screen.blit(bg, (0, 0))
pygame.display.flip()

# Load image
ball_image = pygame.image.load('ball.gif')

# The following lines work around a limitation of the convert method
black = (0, 0, 0, 255)
new_ball_image = pygame.Surface(ball_image.get_size(), 0, screen)
new_ball_image.blit(ball_image, (0, 0))
new_ball_image.set_colorkey(black)
ball_image = new_ball_image
del new_ball_image

# Prepare cache of rotated images
ball_images = {0: ball_image}
ball_rect = ball_image.get_rect()

for angle in xrange(1, 360):

    rotated_image = pygame.transform.rotate(ball_image, angle)
    rotated_rect = rotated_image.get_rect()

    if rotated_rect == ball_rect:

        rotated_image.set_colorkey(black)    
        ball_images[angle] = rotated_image
                
    else:

        # Rotated images are usually larger than their source
        # Trim any new transparent pixels around the edges
        
        new_rotated_image = pygame.Surface(ball_rect.size, 0, screen)

        offset_x = (ball_rect.width - rotated_rect.width) // 2
        offset_y = (ball_rect.height - rotated_rect.height) // 2
        new_rotated_image.blit(rotated_image, (offset_x, offset_y))

        new_rotated_image.set_colorkey(black)    
        ball_images[angle] = new_rotated_image
        
        del new_rotated_image, offset_x, offset_y

del rotated_image, rotated_rect

# Define sprite class
class Ball(pygame.sprite.Sprite):

    """A sprite class for each ball
    
    x and y     starting position for the ball
    speed       pixels per frame
    direction   direction that the ball is going in degrees
    images      dictionary cache of images
    angle       starting rotation of ball image in degrees
    
    """

    def __init__(self, x, y, speed, direction, images, angle, ang_vel):

        pygame.sprite.Sprite.__init__(self)

        self.speed = speed
        self.direction = direction % 360
        self.images = images
        self.angle = angle % 360.0
        self.angular_velocity = ang_vel

        self.images = images
        self.image = images[round(self.angle)]
        self.rect = self.image.get_rect(center=(int(x), int(y)))
    
    def update(self):

        # Update location
        if (self.rect.left < 0 and
            cos(radians(self.direction)) < 0.0) or \
            (self.rect.right >= screen_width and
             cos(radians(self.direction)) > 0.0):
             
            self.direction = (180 - self.direction) % 360

        if (self.rect.top < 0 and
            sin(radians(self.direction)) < 0.0) or \
            (self.rect.bottom >= screen_height and
             sin(radians(self.direction)) > 0.0):

            self.direction = (-self.direction) % 360
    
        dx = self.speed * cos(radians(self.direction)) 
        dy = self.speed * sin(radians(self.direction))   
        
        self.rect.move_ip(dx, dy)
        
        # Update image angle
        self.angle = (self.angle + self.angular_velocity) % 360.0
        self.image = self.images[int(self.angle)]

sprites = pygame.sprite.RenderUpdates(
   Ball(150, 200, 3, 45, ball_images, 45, 0.5),
   Ball(300, 200, 6, 80, ball_images, 200, 1.5),
   Ball(150, 200, 9, 160, ball_images, 5, 2.5),
                                     )

Clock = pygame.time.Clock()
running = True
        
while running:

    # Disable automatic garbage collection
    gc.disable()
    
    # Process events
    for event in pygame.event.get():
        if event.type == QUIT:
            running = False
        elif event.type == KEYDOWN:
            if event.key == K_ESCAPE:
                running = False

    # Draw
    sprites.clear(screen, bg)
    sprites.update()
    dirty = sprites.draw(screen)
    pygame.display.update(dirty)
    
    # Manually collect garbage
    gc.collect()
    
    # Set speed            
    Clock.tick(60)
