"""Senso.py - Main game script
Copyright 2010 Julian Marchant ("onpon4")

This file is part of Senso.

Senso is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Senso is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Senso.  If not, see <http://www.gnu.org/licenses/>.

Start Date:      June 20, 2010
Completion Date: July 12, 2010
Version:         0.2.1
Language:        Python 2.6
Dependencies:    Pygame 1.9

Game description:
    In this game, two teams try to defeat each other by destroying moving
    targets on the other team's side.

    Each team is made up of 1 or more ships. The ships can move in any
    direction and fire a basic gun akin to games like Space Invaders and
    Galaga. Each ship may only have one bullet on the screen at a time.
    When two bullets (one from each team) collide, they annihilate each
    other.

    Each ship has 10 HP. If a ship is hit by an enemy bullet, 1 HP is lost.
    If HP reaches 0, the ship is destroyed.

    At the start of the game, a bomb will be spawned in the center of the
    arena. This bomb can be moved by shooting it. After a set amount of
    time, the bomb will turn red and explode, causing 5 damage to all ships
    in a given radius. A bomb is then spawned a set amount of time later.

    On each side of the screen will be a set number of moving targets, or
    "Senso Turrets". These will be unaffected by bombs, but will be destroyed
    after they are hit by 5 enemy bullets. During the course of the game,
    they will shoot in an attempt to defend themselves.

    There are two ways a player can win: The first, most streatforward
    way is to destroy all Senso Turrets on the opponent's side. The second
    way is to destroy all enemy ships.

"""

VERSION = (0,2,1,1)

import pygame, sys, random, math, os, senso_io as io
from pygame.locals import *

def darken(color, intensity):
    red, green, blue = color
    return round(red*intensity), round(green*intensity), round(blue*intensity)

class Game(object):
    """    The main game class.
    Only one of these should be created.

    There are several arguments to the creation of this class. Here is an
    explanation of all of them:

    screensize: The size of the game screen (i.e., resolution). Should be a
        tuple containing 2 integer values (w,h).

    bgcolor: The color of the main background. Should be a 3-integer tuple
        (R,G,B)
    
    lncolor: The color of the line in the center. Should be a 3-integer tuple
        (R,G,B)

    fps_max: The maximum frame rate. Should be an integer.

    bullet_size: size of all bullets. A tuple with 2 values (w,h)

    controls: A list of "Controls" dictionaries. Each dictionary represents
        controls for an individual ship. The index of the dictionary should
        correspond to the index of a ship. For AI ships, this value is ignored,
        so it is best to simply put ``None``. Dictionaries should have the
        following keys:
            'up' - The Up movement key
            'down' - The Down movement key
            'left' - The Left movement key
            'right' - The Right movement key
            'shoot' - The shoot key

    ships: A list of "Ship" dictionaries, each one representing a single ship.
        Dictionaries should have the following keys:
            'team' - The team the ship is on. Either 0 or 1.
            'human' - Whether the ship is a human. Either True (human
                controlled) or False (AI controlled)
            'position' - The ship's starting position. A tuple containing two
                integer values (x,y)
            'size' - The ship's size. A tuple containing two integer values
                (width,height)
            'speed' - The speed of the ship. An integer representing speed in
                pixels per second.
            'color' - The color of the ship. A tuple containing 3 values
                (red,green,blue)

    targets: A list of "Target" dictionaries, each one representing a single
        target (Senso Unit). Dictionaries should have the following keys:
            'team' - The team the target is on. Either 0 or 1.
            'position' - The target's starting position. A tuple containing two
                integer values (x,y)
            'size' - The target's size. A tuple containing two integer values
                (width,height)
            'xspeed' - the target's initial horizontal speed. An integer
                representing speed in pixels per second.
            'yspeed' - The target's initial vertical speed. An integer
                representing speed in pixels per second.
            'color1' - The target's back color (the color of the square).
                A tuple containing 3 values (red,green,blue)
            'color2' - The target's front color (the color of the circle).
                A tuple containing 3 values (red,green,blue)
            'shootwait' - The amount of time (milliseconds) this target waits
                before trying to shoot. An integer.
            'shootfreq' - The chance (out of 1) that the target will actually
                shoot when it attempts to. A decimal.
    
    """
    def __init__(self, screensize, bgcolor, lncolor, fps_max, bullet_size, \
                 controls, ships, targets):
        print('Starting Pygame...')
        pygame.init()
        random.seed()
        
        self.rect = Rect((0,0), screensize)
        self.arena = {0:Rect(0,0,screensize[0]//2,screensize[1]), \
                      1:Rect(screensize[0]//2,0,screensize[0]//2,screensize[1])}

        print('Initializing variables...')
        #controls
        self.ctr_up = {}
        self.ctr_down = {}
        self.ctr_left = {}
        self.ctr_right = {}
        self.ctr_shoot = {}
        
        #Game variables
        self.fps_max = fps_max
        self.time_passed = 0
        self.bullet_size = bullet_size

        print('Setting up window...')
        #Window
        self.window = pygame.display.set_mode(self.rect.size)
        pygame.display.set_caption('Senso')

        print('Creating background...')
        #Background
        self.background = pygame.Surface(self.rect.size).convert()
        self.background.fill(bgcolor)
        pygame.draw.line(self.background, lncolor, \
                         (self.rect.size[0]//2,0), \
                         (self.rect.size[0]//2,self.rect.size[1]), 3)
        self.window.blit(self.background, (0,0))
        pygame.display.update()

        print('Managing game objects...')
        #Events
        pygame.event.set_allowed([QUIT, KEYDOWN, KEYUP])
        
        #Clock
        self.clock = pygame.time.Clock()
        
        #Groups/sprites
        self.all = pygame.sprite.RenderUpdates()

        print('Creating players...')
        self.ships = {}
        numships = 0
        for i in ships: numships += 1
        for i in xrange(0,numships):
            shp = Ship(game=self, team=ships[i]['team'], human=ships[i]['human'], \
                       position=ships[i]['position'], size=ships[i]['size'], \
                       speed=ships[i]['speed'], color=ships[i]['color'])
            self.ships[id(shp)] = shp
            self.all.add(shp)
            if shp.human:
                self.ctr_up[id(shp)] = controls[i]['up']
                self.ctr_down[id(shp)] = controls[i]['down']
                self.ctr_left[id(shp)] = controls[i]['left']
                self.ctr_right[id(shp)] = controls[i]['right']
                self.ctr_shoot[id(shp)] = controls[i]['shoot']

        print('Creating Senso Turrets...')
        self.targets = {}
        numtargets = 0
        for i in targets: numtargets += 1
        for i in xrange(0,numtargets):
            sen = Target(game=self, team=targets[i]['team'], \
                         position=targets[i]['position'], \
                         xspeed=targets[i]['xspeed'], \
                         yspeed=targets[i]['yspeed'], size=targets[i]['size'], \
                         color1=targets[i]['color1'], \
                         color2=targets[i]['color2'], \
                         shoot_wait=targets[i]['shootwait'], \
                         shoot_freq=targets[i]['shootfreq'])
            self.targets[id(sen)] = sen
            self.all.add(sen)

        self.bullets = {} #No bullets will be created at the start of the game
        
        print('Starting game...')
        self.run()

    def run(self):
        while 1:
            #events
            for event in pygame.event.get():
                if event.type == QUIT \
                   or (event.type == KEYDOWN and event.key == K_ESCAPE):
                    self.quit()

                elif event.type == KEYDOWN:
                    for i in self.ships.keys():
                        if self.ships[i].human:
                            if event.key == self.ctr_up[i]:
                                self.ships[i].set_motion(0,-1)
                            elif event.key == self.ctr_down[i]:
                                self.ships[i].set_motion(0,1)
                            elif event.key == self.ctr_left[i]:
                                self.ships[i].set_motion(-1,0)
                            elif event.key == self.ctr_right[i]:
                                self.ships[i].set_motion(1,0)
                            elif event.key == self.ctr_shoot[i]:
                                self.ships[i].shoot()
                
                elif event.type == KEYUP:
                    for i in self.ships.keys():
                        if self.ships[i].human:
                            if event.key == self.ctr_up[i]:
                                self.ships[i].set_motion(0,1)
                            elif event.key == self.ctr_down[i]:
                                self.ships[i].set_motion(0,-1)
                            elif event.key == self.ctr_left[i]:
                                self.ships[i].set_motion(1,0)
                            elif event.key == self.ctr_right[i]:
                                self.ships[i].set_motion(-1,0)

            self.time_passed = self.clock.tick(self.fps_max) / 1000.

            pygame.display.set_caption('Senso ({0} FPS)'.format(int(round(self.clock.get_fps()))))

            #Redraw
            self.all.clear(self.window, self.background)
            self.all.update()
            dirty = self.all.draw(self.window)
            pygame.display.update(dirty)

    def quit(self):
        pygame.quit()
        sys.exit()

class Ship(pygame.sprite.Sprite):
    def __init__(self, game, team, human, position, size, speed, color):
        pygame.sprite.Sprite.__init__(self)

        #Initialize vars
        self.x, self.y = position
        self.rect = Rect((self.x-(size[0]//2),self.y-(size[1]//2)),size)
        self.game = game
        self.human = human
        self.team = team
        self.speed = speed
        self.direction = (not team) - team
        self.color = color
        self.xspeed = 0
        self.yspeed = 0
        self.health = 10
        self.can_shoot = 1

        #Image
        self.image = pygame.Surface(size)
        transcolor = (255-color[0],255-color[1],255-color[2])
        self.image.fill(transcolor)
        pygame.draw.polygon(self.image, color, \
                            [(0,0), (0,size[1]), (size[0],size[1]//2)])
        self.image.set_colorkey(transcolor)
        self.image = pygame.transform.flip(self.image, self.team, 0)

    def update(self):
        if not self.human:
            #AI control
            pass

        #Move
        self.x += self.xspeed * self.game.time_passed
        self.y += self.yspeed * self.game.time_passed
        
        self.rect.centerx = round(self.x)
        self.rect.centery = round(self.y)

        #Stay inside the window
        if self.rect.top < self.game.arena[self.team].top:
            self.rect.top = self.game.arena[self.team].top
            self.x = self.rect.centerx
            self.y = self.rect.centery
        if self.rect.bottom > self.game.arena[self.team].bottom:
            self.rect.bottom = self.game.arena[self.team].bottom
            self.x = self.rect.centerx
            self.y = self.rect.centery
        if self.rect.left < self.game.arena[self.team].left:
            self.rect.left = self.game.arena[self.team].left
            self.x = self.rect.centerx
            self.y = self.rect.centery
        if self.rect.right > self.game.arena[self.team].right:
            self.rect.right = self.game.arena[self.team].right
            self.x = self.rect.centerx
            self.y = self.rect.centery

    def set_motion(self, xmove, ymove):
        self.xspeed += xmove*self.speed
        self.yspeed += ymove*self.speed

    def reset_motion(self):
        self.xspeed = 0
        self.yspeed = 0

    def shoot(self):
        if self.can_shoot:
            
            if self.direction == 1:
                bulletpos = self.rect.midright
            else:
                bulletpos = self.rect.midleft

            bullet = Bullet(shooter=self, team=self.team, \
                            position=bulletpos, size=self.game.bullet_size, \
                            speed=500, direction=self.direction, \
                            color=self.color)

            self.game.bullets[id(bullet)] = bullet

            self.game.all.add(bullet)
            self.can_shoot = 0

    def hurt(self, damage):
        self.health -= damage
        if self.health <= 0:
            self.destroy()
        else:
            self.image.fill(self.image.get_colorkey())
            pygame.draw.polygon(self.image, darken(self.color,self.health/10.), \
                                [(0,0), (0,self.rect.size[1]), \
                                (self.rect.size[0],self.rect.size[1]//2)])
            self.image = pygame.transform.flip(self.image,self.team,0)

    def destroy(self):
        self.kill()

        endgame = 1
        for i in self.game.ships.keys():
            if self.game.ships[i] != self \
               and self.game.ships[i].team == self.team:
                endgame = 0
                break

        if endgame:
            print 'Team {0} wins!'.format((not self.team) + 1)
            self.game.quit()
        else:
            del self.game.ships[id(self)]

class Bullet(pygame.sprite.Sprite):
    def __init__(self, shooter, team, position, size, speed, direction, color):
        pygame.sprite.Sprite.__init__(self)

        #vars
        self.x, self.y = position
        self.rect = Rect(position,size)
        self.shooter = shooter
        self.game = self.shooter.game
        self.team = team
        self.xspeed = direction*speed
        self.yspeed = 0
        self.direction = direction

        #Image
        self.image = pygame.Surface(size)
        self.image.fill(color)

    def update(self):
        self.x += self.xspeed * self.game.time_passed
        self.rect.centerx = round(self.x)

        #Collision detection
        if self.rect.top < self.shooter.game.rect.top \
           or self.rect.bottom > self.shooter.game.rect.bottom \
           or self.rect.left < self.shooter.game.rect.left \
           or self.rect.right > self.shooter.game.rect.right:
            self.destroy()
        
        for i in self.shooter.game.ships.keys():
            if self.shooter.game.ships[i].team != self.team \
               and self.rect.colliderect(self.shooter.game.ships[i].rect):
                self.shooter.game.ships[i].hurt(1)
                self.destroy()
                break

        for i in self.shooter.game.bullets.keys():
            if self.shooter.game.bullets[i].team != self.team \
               and self.rect.colliderect(self.shooter.game.bullets[i].rect):
                self.shooter.game.bullets[i].destroy()
                self.destroy()
                break

        for i in self.shooter.game.targets.keys():
            if self.shooter.game.targets[i].team != self.team \
               and self.rect.colliderect(self.shooter.game.targets[i].rect):
                self.shooter.game.targets[i].hurt(1)
                self.destroy()
                break

    def destroy(self):
        self.kill()
        self.shooter.can_shoot = 1
        del self.shooter.game.bullets[id(self)]

class Bomb(pygame.sprite.Sprite):
    def __init__(self, game, position, friction, radius, time, color1, color2):
        pygame.sprite.Sprite.__init__(self)

        self.x, self.y = position

        #vars
        self.rect = Rect((self.x-radius,self.y-radius), (2*radius,2*radius))
        self.game = game
        self.xspeed = 0
        self.yspeed = 0
        self.friction = friction
        self.radius = radius
        self.time = time
        self.time_remaining = self.time
        self.clock = pygame.time.Clock()
        self.color1 = color1
        self.color2 = color2

        #Image
        self.image = pygame.surface(self.rect.size)
        transcolor = (255-color1[0],255-color1[1],255-color1[2])
        self.image.fill(transcolor)
        pygame.draw.circle(self.image, color1, position, radius)
        self.image.set_colorkey(transcolor)

    def update(self):
        self.x += self.xspeed * self.game.time_passed
        self.y += self.yspeed * self.game.time_passed
        
        self.rect.centerx = self.x
        self.rect.centery = self.y

        #Bounce off edges
        if self.rect.top < self.game.rect.top:
            self.rect.top = self.game.rect.top
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.yspeed *= -1
        if self.rect.bottom > self.game.rect.bottom:
            self.rect.bottom = self.game.rect.bottom
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.yspeed *= -1
        if self.rect.left < self.game.rect.left:
            self.rect.left = self.game.rect.left
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.xspeed *= -1
        if self.rect.right > self.game.rect.right:
            self.rect.right = self.game.rect.right
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.xspeed *= -1

        #Friction
        if self.xspeed != 0:
            if abs(self.xspeed) <= self.friction:
                self.xspeed = 0
            else:
                self.xspeed -= (self.xspeed//abs(self.xspeed)) * friction
        if self.yspeed != 0:
            if abs(self.yspeed) <= self.friction:
                self.yspeed = 0
            else:
                self.yspeed -= (self.yspeed//abs(self.yspeed)) * friction

        #Bomb
        self.time_remaining -= self.clock.tick()
        if self.time_remaining <= 0:
            self.destroy()
        elif self.time_remaining <= 1000:
            self.image.fill(self.image.get_colorkey)
            pygame.draw.circle(self.image, self.color2, \
                               (self.rect.size[0]//2,self.rect.size[1]//2), \
                               self.radius)
            
    def destroy(self):
        pass

class Target(pygame.sprite.Sprite):
    def __init__(self, game, team, position, xspeed, yspeed, size, \
                 color1, color2, shoot_wait=1000, shoot_freq=.1):
        pygame.sprite.Sprite.__init__(self)

        self.x, self.y = position
        self.c_radius = int(min(round(size[0]*0.8),round(size[1]*0.8)) // 2)

        #vars
        self.rect = Rect((self.x-(size[0]//2),self.y-(size[1]//2)), size)
        self.game = game
        self.team = team
        self.xspeed = xspeed
        self.yspeed = yspeed
        self.health = 5
        self.direction = (not team) - team
        self.shoot_wait = shoot_wait
        self.clock = pygame.time.Clock()
        self.wait_time = self.shoot_wait
        self.shoot_freq = shoot_freq
        self.color1 = color1
        self.color2 = color2

        #Image
        self.image = pygame.Surface(self.rect.size)
        self.image.fill(self.color1)
        pygame.draw.circle(self.image, self.color2, \
                           (self.rect.size[0]//2,self.rect.size[1]//2), \
                           self.c_radius)

    def update(self):
        self.x += self.xspeed * self.game.time_passed
        self.y += self.yspeed * self.game.time_passed
        
        self.rect.centerx = round(self.x)
        self.rect.centery = round(self.y)

        #Bounce off edges
        if self.rect.top < self.game.arena[self.team].top:
            self.rect.top = self.game.arena[self.team].top
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.yspeed *= -1
        if self.rect.bottom > self.game.arena[self.team].bottom:
            self.rect.bottom = self.game.arena[self.team].bottom
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.yspeed *= -1
        if self.rect.left < self.game.arena[self.team].left:
            self.rect.left = self.game.arena[self.team].left
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.xspeed *= -1
        if self.rect.right > self.game.arena[self.team].right:
            self.rect.right = self.game.arena[self.team].right
            self.x = self.rect.centerx
            self.y = self.rect.centery
            self.xspeed *= -1

        #Shoot in defense
        self.wait_time -= self.clock.tick()

        if self.wait_time <= 0:
            self.wait_time = self.shoot_wait
            if random.random() < self.shoot_freq:
                self.shoot()

    def shoot(self):
        if self.direction == 1:
            bulletpos = self.rect.midright
        else:
            bulletpos = self.rect.midleft

        bullet = Bullet(shooter=self, team=self.team, \
                        position=bulletpos, size=self.game.bullet_size, \
                        speed=500, direction=self.direction, \
                        color=self.color2)

        self.game.bullets[id(bullet)] = bullet

        self.game.all.add(bullet)

    def hurt(self, damage):
        self.health -= damage
        if self.health <= 0:
            self.destroy()
        else:
            self.image.fill(darken(self.color1,self.health/5.))
            pygame.draw.circle(self.image, darken(self.color2,self.health/5.), \
                               (self.rect.size[0]//2,self.rect.size[1]//2), \
                               self.c_radius)

    def destroy(self):
        self.kill()

        endgame = 1
        for i in self.game.targets.keys():
            if self.game.targets[i] != self \
               and self.game.targets[i].team == self.team:
                endgame = 0
                break

        if endgame:
            print 'Team {0} wins!'.format((not self.team) + 1)
            self.game.quit()
        else:
            del self.game.targets[id(self)]

if __name__ == '__main__':
    ctrls = io.read('controls.snf')['controls']
    games = io.read('games.snf')['game']

    print 'Please enter the number for the game you want to play. The games'
    print 'available are:'

    ngames = 0
    for i in games: ngames += 1
    for i in xrange(0,ngames):
        print '{0} - {1}'.format(i, games[i]['name'])

    e = 1
    while e:
        e = 0
        try:
            gamechoice = games[int(raw_input())]
        except ValueError:
            print 'Error: You must enter a numeric (integer) value.'
            e = 1
        except IndexError:
            print 'Error: Index out of range. Please enter the index of one of'
            print 'the games listed above.'
            e = 1

    selectedgame = io.read(os.path.join('games', gamechoice['file']))
    settings = selectedgame['settings'][0]

    try:
        shps = selectedgame['ship']
    except KeyError:
        shps = []
    try:
        sensos = selectedgame['target']
    except KeyError:
        sensos = []

    game = Game(settings['screensize'], settings['bgcolor'], \
                settings['lncolor'], settings['fps'], settings['bulletsize'], \
                ctrls, shps, sensos)
