// CoMET - The Crimson Fields Map Editing Tool
// Copyright (C) 2002-2004 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// eventwindow.cpp
///////////////////////////////////////////////////////////////////////

#include "eventwindow.h"
#include "extwindow.h"
#include "extwindow2.h"
#include "uiaux.h"

extern const char *player2_labels[];

// initialize static class variables
const short EdEventGenericWindow::B_ID_TITLE = 1;
const short EdEventGenericWindow::B_ID_MSG = 2;
const short EdEventGenericWindow::B_ID_PLAYER = 3;
const short EdEventGenericWindow::B_ID_DEPEND = 4;
const short EdEventGenericWindow::B_ID_TRIGGER = 5;
const short EdEventGenericWindow::B_ID_EVENT = 6;
const short EdEventGenericWindow::B_ID_TITLE_OK = 7;
const short EdEventGenericWindow::B_ID_MSG_OK = 8;
const short EdEventGenericWindow::B_ID_DEPEND_OK = 9;

const short EdEventCreateUnitWindow::B_ID_UNIT = 1;
const short EdEventCreateUnitWindow::B_ID_UNIT_OK = 2;
const short EdEventCreateUnitWindow::B_ID_SHOP = 3;
const short EdEventCreateUnitWindow::B_ID_SHOP_OK = 4;

const short EdEventManipEventWindow::B_ID_EVENT = 1;
const short EdEventManipEventWindow::B_ID_EVENT_OK = 2;
const short EdEventManipEventWindow::B_ID_ACTION = 3;
const short EdEventManipEventWindow::B_ID_DISABLED = 4;

const short EdEventMiningWindow::B_ID_MINE = 1;
const short EdEventMiningWindow::B_ID_MINE_OK = 2;
const short EdEventMiningWindow::B_ID_ACTION = 3;
const short EdEventMiningWindow::N_ID_CRYSTALS = 4;

const short EdEventNextMapWindow::B_ID_MAP = 1;
const short EdEventNextMapWindow::B_ID_MAP_OK = 2;

const short EdEventResearchWindow::B_ID_UNIT = 1;
const short EdEventResearchWindow::B_ID_UNIT_OK = 2;
const short EdEventResearchWindow::B_ID_SHOP = 3;
const short EdEventResearchWindow::B_ID_SHOP_OK = 4;

const short EdEventScoreWindow::B_ID_MSG = 1;
const short EdEventScoreWindow::B_ID_MSG_OK = 2;
const short EdEventScoreWindow::B_ID_TITLE = 3;
const short EdEventScoreWindow::B_ID_TITLE_OK = 4;
const short EdEventScoreWindow::N_ID_SCORE = 5;

const short EdTrigHaveShopWindow::B_ID_SHOP = 1;
const short EdTrigHaveShopWindow::B_ID_SHOP_OK = 2;
const short EdTrigHaveShopWindow::B_ID_PLAYER = 3;
const short EdTrigHaveShopWindow::N_ID_TURN = 4;

const short EdTrigHaveUnitWindow::B_ID_UNIT = 1;
const short EdTrigHaveUnitWindow::B_ID_UNIT_OK = 2;
const short EdTrigHaveUnitWindow::B_ID_PLAYER = 3;
const short EdTrigHaveUnitWindow::N_ID_TURN = 4;

const short EdTrigUnitDestroyedWindow::B_ID_UNIT = 1;
const short EdTrigUnitDestroyedWindow::B_ID_UNIT_OK = 2;
const short EdTrigUnitDestroyedWindow::B_ID_PLAYER = 3;

const short EdTrigUnitPositionWindow::B_ID_UNIT = 1;
const short EdTrigUnitPositionWindow::B_ID_UNIT_OK = 2;
const short EdTrigUnitPositionWindow::N_ID_POSX = 3;
const short EdTrigUnitPositionWindow::N_ID_POSY = 4;


////////////////////////////////////////////////////////////////////////
// NAME       : EdEventGenericWindow::EdEventGenericWindow
// DESCRIPTION: This window allows modification of the fields common to
//              all event types, and also offers links to event-specific
//              options.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdEventGenericWindow::EdEventGenericWindow( Event &event,
       Mission &mission, View *view ) :
       Window(WIN_CENTER, view), event(event), mission(mission) { 
  // calculate dimensions
  SetSize( MIN(sfont->Width() * 35 + 20, view->Width()),
           MIN(sfont->Height() * 11 + 110, view->Height()) );

  unsigned short wdh = sfont->Height() + 8, wdx, wdy = 5;
  Widget *wd;
  char buf[48] = "";

  wd = new NumberWidget( 0, sfont->Width() * 2 + 10, wdy,
       sfont->Width() * 6, wdh, event.ID(),
       0, 32767, WIDGET_STR_CONST|WIDGET_ALIGN_LEFT, "ID", this );

  wdx = wd->LeftEdge() + wd->Width() + sfont->Width() * 6 + 10;
  wd = new CycleWidget( B_ID_PLAYER, wdx, wdy, w - wdx - 5, wdh,
       0, "_Player", event.Player(), player2_labels, this );
  wd->SetHook( this );

  wdy += wd->Height() + 5;
  wd = new ButtonWidget( B_ID_TITLE, 5, wdy,
       sfont->Width() * 9 + 10, wdh, 0, "Msg _Title", this );
  wd->SetHook( this );

  wdx = wd->LeftEdge() + wd->Width() + 5;
  e_title = new StringWidget( 0, wdx, wdy, w - wdx - 5, wdh,
            (mission.GetMessage(event.Title()) ?
             mission.GetMessage(event.Title()) : ""),
            30, WIDGET_STR_CONST, NULL, this );

  wdy += wd->Height() + 5;
  wd = new ButtonWidget( B_ID_MSG, 5, wdy, w - 10, wdh,
       0, "Set Event _Message", this );
  wd->SetHook( this );

  wdy += wd->Height() + 5;
  e_msg = new TextScrollWidget( 0, 5, wdy, w - 10, sfont->Height() * 5,
          (mission.GetMessage(event.Message()) ?
           mission.GetMessage(event.Message()) : ""),
          0, NULL, this );

  wdy += e_msg->Height() + 5;
  wd = new ButtonWidget( B_ID_TRIGGER, 5, wdy,
       sfont->Width() * 12 + 10, wdh, 0,
       "Edit T_rigger", this );
  wd->SetHook( this );

  wdy += wd->Height() + 5;
  wd = new ButtonWidget( B_ID_EVENT, 5, wdy, wd->Width(), wdh,
       (event.Type() == EVENT_MESSAGE) ? WIDGET_DISABLED : 0,
       "Edit _Event", this );
  wd->SetHook( this );

  wdy += wd->Height() + 5;
  wd = new ButtonWidget( B_ID_DEPEND, 5, wdy, wd->Width(), wdh,
       0, "_Depends on", this );
  wd->SetHook( this );

  wdx = wd->LeftEdge() + wd->Width() + 5;
  if ( event.Dependency() >= 0 ) {
    Event *e = mission.GetEventByID( event.Dependency() );
    if ( e ) sprintf( buf, "%s (%d)", e->Name(), e->ID() );
  }
  e_depends = new StringWidget( 0, wdx, wdy, w - wdx - 5, wdh,
              buf, 47, WIDGET_STR_CONST, NULL, this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventGenericWindow::Draw
// DESCRIPTION: Draw the window.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdEventGenericWindow::Draw( void ) {
  Window::Draw();

  unsigned short lblx = sfont->Width() * 12 + 20,
                 lbly = e_msg->TopEdge() + e_msg->Height() + 9;

  sfont->Write( ':', this, lblx, lbly );
  sfont->Write( event.TriggerName(), this, lblx + sfont->Width() * 2, lbly );

  lbly += sfont->Height() + 13;
  sfont->Write( ':', this, lblx, lbly );
  sfont->Write( event.Name(), this, lblx + sfont->Width() * 2, lbly );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventGenericWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the button
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdEventGenericWindow::WidgetActivated( Widget *widget, Window *win ) {
  TLWNode *n;

  switch ( widget->ID() ) {
  case B_ID_TITLE:
    TLWListBuilder::BuildMsgList( mission.GetMessages(), tlist );
    new ListSelectWindow( "Select event title", tlist, event.Title(),
                          true, this, B_ID_TITLE_OK, view );
    break;

  case B_ID_TITLE_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetTitle( n ? n->ID() : -1 );
    e_title->SetString( n ? n->Name() : NULL, true );
    tlist.Clear();
    break;

  case B_ID_MSG:
    TLWListBuilder::BuildMsgList( mission.GetMessages(), tlist );
    new ListSelectWindow( "Select event message", tlist, event.Message(),
                          true, this, B_ID_MSG_OK, view );
    break;

  case B_ID_MSG_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetMessage( n ? n->ID() : -1 );
    e_msg->SetText( n ? mission.GetMessage(n->ID()) : NULL );
    tlist.Clear();
    break;

  case B_ID_DEPEND:
    TLWListBuilder::BuildEventList( mission.GetEvents(), tlist );
    new ListSelectWindow( "Select event", tlist, event.Dependency(),
                          true, this, B_ID_DEPEND_OK, view );
    break;

  case B_ID_DEPEND_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetDependency( n ? n->ID() : -1 );
    e_depends->SetString( n ? n->Name() : NULL );
    tlist.Clear();
    break;

  case B_ID_TRIGGER:
    switch ( event.Trigger() ) {
    case ETRIGGER_HAVE_BUILDING:
      new EdTrigHaveShopWindow( event, mission, view ); break;
    case ETRIGGER_HAVE_UNIT:
      new EdTrigHaveUnitWindow( event, mission, view ); break;
    case ETRIGGER_TURN:
      new EdTrigTurnWindow( event, view ); break;
    case ETRIGGER_UNIT_DESTROYED:
      new EdTrigUnitDestroyedWindow( event, mission, view ); break;
    case ETRIGGER_UNIT_POSITION:
      new EdTrigUnitPositionWindow( event, mission, view ); break;
    default:
      new NoteWindow("Sorry", "Not yet implemented", 0, view);
    }
    break;

  case B_ID_EVENT:
    switch ( event.Type() ) {
    case EVENT_CREATE_UNIT:
      new EdEventCreateUnitWindow( event, mission, view ); break;
    case EVENT_MANIPULATE_EVENT:
      new EdEventManipEventWindow( event, mission, view ); break;
    case EVENT_MINING:
      new EdEventMiningWindow( event, mission, view ); break;
    case EVENT_NEXT_MAP:
      new EdEventNextMapWindow( event, mission, view ); break;
    case EVENT_RESEARCH:
      new EdEventResearchWindow( event, mission, view ); break;
    case EVENT_SCORE:
      new EdEventScoreWindow( event, mission, view ); break;
    default:
      new NoteWindow("Sorry", "Not yet implemented", 0, view);
    }
    break;
  case B_ID_PLAYER:
    event.SetPlayer( static_cast<CycleWidget *>(widget)->GetValue() );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventCreateUnitWindow::EdEventCreateUnitWindow
// DESCRIPTION: This window allows configuration of a CREATE_UNIT
//              event.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdEventCreateUnitWindow::EdEventCreateUnitWindow( Event &event,
                         Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 30 + 15, view->Width()),
           sfont->Height() * 3 + 40 );

  Widget *wd;
  unsigned short wdx, wdh = sfont->Height() + 8;

  wd = new ButtonWidget( B_ID_UNIT, 5, 5,
       sfont->Width() * 4 + 10, wdh, 0, "_Unit", this );
  wd->SetHook( this );

  wdx = wd->LeftEdge() + wd->Width() + 5;
  e_unit = new StringWidget( 0, wdx, 5, w - wdx - 5, wdh,
           mission.GetUnitSet().GetUnitInfo(event.GetData(0))->Name(),
           30, WIDGET_STR_CONST, NULL, this );

  wd = new ButtonWidget( B_ID_SHOP, 5, wd->TopEdge() + wdh + 5,
       sfont->Width() * 4 + 10, wdh, 0, "_Shop", this );
  wd->SetHook( this );

  Building *b = mission.GetBuildingByID( event.GetData(1) );
  e_shop = new StringWidget( 0, wdx, wd->TopEdge(),
           e_unit->Width(), wdh, b ? b->Name() : "Error",
           30, WIDGET_STR_CONST, NULL, this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventCreateUnitWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdEventCreateUnitWindow::WidgetActivated( Widget *widget, Window *win ) {
  TLWNode *n;

  switch( widget->ID() ) {
  case B_ID_UNIT:
    TLWListBuilder::BuildUnitTypesList( mission.GetUnitSet(), tlist );
    new ListSelectWindow( "Select unit", tlist,
        event.GetData(0), false, this, B_ID_UNIT_OK, view );
    break;
  case B_ID_UNIT_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 0, n ? n->ID() : -1 );
    e_unit->SetString( n ? n->Name() : "Error" );
    tlist.Clear();
    break;
  case B_ID_SHOP:
    TLWListBuilder::BuildShopList( mission.GetBuildings(), tlist );
    new ListSelectWindow( "Select shop", tlist,
        event.GetData(1), false, this, B_ID_SHOP_OK, view );
    break;
  case B_ID_SHOP_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 1, n ? n->ID() : -1 );
    e_shop->SetString( n ? n->Name() : "Error" );
    tlist.Clear();
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventManipEventWindow::EdEventManipEventWindow
// DESCRIPTION: This window allows configuration of a MANIPULATE_EVENT
//              event.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdEventManipEventWindow::EdEventManipEventWindow( Event &event,
                         Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  static const char *action_labels[] = { "Set flags",
                                         "Clear flags",
                                         "Toggle flags", 0 };

  SetSize( MIN(sfont->Width() * 25 + 15, view->Width()),
           sfont->Height() * 4 + 50 );

  Widget *wd;
  unsigned short wdx, wdy, wdh = sfont->Height() + 8;
  char buf[48] = "Error";

  wd = new ButtonWidget( B_ID_EVENT, 5, 5,
       sfont->Width() * 5 + 10, wdh, 0, "_Event", this );
  wd->SetHook( this );

  Event *e = mission.GetEventByID( event.GetData(0) );
  if ( e ) sprintf( buf, "%s (%d)", e->Name(), e->ID() );
  wdx = wd->LeftEdge() + wd->Width() + 5;
  e_event = new StringWidget( 0, wdx, 5, w - wdx - 5, wdh,
            buf, 30, WIDGET_STR_CONST, NULL, this );

  wdx = sfont->Width() * 6 + 10;
  wdy = wdh + 10;
  wd = new CycleWidget( B_ID_ACTION, wdx, wdy,
       w - wdx - 5, wdh, 0, "_Action",
       event.GetData(2), action_labels, this );
  wd->SetHook( this );

  wdx = w - 10 - DEFAULT_CBW_SIZE;
  wdy += wdh + 5;
  wd = new CheckboxWidget( B_ID_DISABLED, wdx, wdy,
       DEFAULT_CBW_SIZE, DEFAULT_CBW_SIZE,
       event.GetData(1) & EFLAG_DISABLED,
       WIDGET_STYLE_GFX|WIDGET_STYLE_NOBORDER|WIDGET_ALIGN_LEFT,
       "_Disabled", this );
  wd->SetHook( this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();

  TLWListBuilder::BuildEventList( mission.GetEvents(), tlist );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventManipEventWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdEventManipEventWindow::WidgetActivated( Widget *widget, Window *win ) {

  switch( widget->ID() ) {
  case B_ID_EVENT:
    new ListSelectWindow( "Select event", tlist,
        event.GetData(0), false, this, B_ID_EVENT_OK, view );
    break;
  case B_ID_EVENT_OK: {
    TLWNode *n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 0, n ? n->ID() : -1 );
    e_event->SetString( n ? n->Name() : "Error" );
    break; }
  case B_ID_ACTION:
    event.SetData( 2, static_cast<CycleWidget *>(widget)->GetValue() );
    break;
  case B_ID_DISABLED:
    event.SetData( 1, event.GetData(1)^EFLAG_DISABLED );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventMiningWindow::EdEventMiningWindow
// DESCRIPTION: This window allows configuration of a MINING event.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdEventMiningWindow::EdEventMiningWindow( Event &event,
                         Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  static const char *action_labels[] = { "Set supplies",
                                         "Modify supplies",
                                         "Set production",
                                         "Modify production", 0 };

  SetSize( MIN(sfont->Width() * 30 + 15, view->Width()),
           sfont->Height() * 4 + 50 );

  Widget *wd;
  unsigned short wdx, wdy, wdh = sfont->Height() + 8;

  wd = new ButtonWidget( B_ID_MINE, 5, 5,
       sfont->Width() * 6, wdh, 0, "_Mine", this );
  wd->SetHook( this );

  wdx = wd->LeftEdge() + wd->Width() + 5;
  Building *b = mission.GetBuildingByID( event.GetData(0) );
  e_mine = new StringWidget( 0, wdx, wd->TopEdge(),
           w - wdx - 5, wdh, b ? b->Name() : "Error",
           30, WIDGET_STR_CONST, NULL, this );

  wdy = wd->Height() + 10;
  wd = new CycleWidget( B_ID_ACTION, wdx, wdy,
       e_mine->Width(), wdh, 0, "_Action",
       event.GetData(2), action_labels, this );
  wd->SetHook( this );

  wdy += wdh + 5;
  e_crystals = new NumberWidget( N_ID_CRYSTALS,
       wdx, wdy, wd->Width(), wdh,
       event.GetData(1), -1000, 10000,
       WIDGET_ALIGN_LEFT, "Amou_nt", this );
  e_crystals->SetHook( this );

  // this is kind of a hack: it's not (yet) possible to
  // change the maximum string length after creation of
  // the widget, so we assumed the maximum above and set
  // the real values now
  SetMiningRange( event.GetData(1) );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventMiningWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdEventMiningWindow::WidgetActivated( Widget *widget, Window *win ) {
  TLWNode *n;

  switch( widget->ID() ) {
  case B_ID_MINE:
    TLWListBuilder::BuildShopList( mission.GetBuildings(), tlist );
    new ListSelectWindow( "Select mine", tlist,
        event.GetData(0), false, this, B_ID_MINE_OK, view );
    break;
  case B_ID_MINE_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 0, n ? n->ID() : -1 );
    e_mine->SetString( n ? n->Name() : "Error" );
    tlist.Clear();
    break;
  case B_ID_ACTION:
    event.SetData( 2, static_cast<CycleWidget *>(widget)->GetValue() );
    SetMiningRange( static_cast<CycleWidget *>(widget)->GetValue() );
    // fall through in case range has changed the number
  case N_ID_CRYSTALS:
    event.SetData( 1, e_crystals->Number() );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventMiningWindow::SetMiningRange
// DESCRIPTION: The minimum and maximum numbers allowed in the "amount"
//              widget are subject of the "action" chosen. We set the
//              correct range here.
// PARAMETERS : action - action to set the range for
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void EdEventMiningWindow::SetMiningRange( short action ) const {
  short min = 0, max = 0;
  switch ( action ) {
  case 0: max = 1000; break;              // set supplies
  case 1: min = -1000; max = 1000; break; // modify supplies
  case 2: max = 200; break;               // set production
  case 3: min = -200; max = 200; break;   // modify production;
  }
  e_crystals->SetMin( min );
  e_crystals->SetMax( max );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventNextMapWindow::EdEventNextMapWindow
// DESCRIPTION: This window allows configuration of a NEXT_MAP event.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdEventNextMapWindow::EdEventNextMapWindow( Event &event,
                      Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 20 + 15, view->Width()),
           sfont->Height() * 2 + 30 );

  unsigned short wdh = sfont->Height() + 8, wdx;
  Widget *wd = new ButtonWidget( B_ID_MAP, 5, 5,
       sfont->Width() * 7 + 10, wdh, 0, "_Mission", this );
  wd->SetHook( this );

  wdx = wd->LeftEdge() + wd->Width() + 5;
  e_map = new StringWidget( 0, wdx, wd->TopEdge(),
           w - wdx - 5, wdh,
           mission.GetMessage(event.GetData(0)) ?
           mission.GetMessage(event.GetData(0)) : "None",
           30, WIDGET_STR_CONST, NULL, this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventNextMapWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdEventNextMapWindow::WidgetActivated( Widget *widget, Window *win ) {
  switch( widget->ID() ) {
  case B_ID_MAP:
    TLWListBuilder::BuildMsgList( mission.GetMessages(), tlist );
    new ListSelectWindow( "Select map name", tlist,
        event.GetData(0), false, this, B_ID_MAP_OK, view );
    break;
  case B_ID_MAP_OK: {
    TLWNode *n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 0, n ? n->ID() : -1 );
    e_map->SetString( n ? n->Name() : "Error" );
    tlist.Clear(); }
  }
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventResearchWindow::EdEventResearchWindow
// DESCRIPTION: This window allows configuration of a RESEARCH event.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdEventResearchWindow::EdEventResearchWindow( Event &event,
                       Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 30 + 15, view->Width()),
           sfont->Height() * 3 + 40 );

  Widget *wd;
  unsigned short wdx, wdh = sfont->Height() + 8;

  wd = new ButtonWidget( B_ID_SHOP, 5, 5,
       sfont->Width() * 4 + 10, wdh, 0, "_Shop", this );
  wd->SetHook( this );

  Building *b = mission.GetBuildingByID( event.GetData(0) );
  wdx = wd->LeftEdge() + wd->Width() + 5;
  e_shop = new StringWidget( 0, wdx, 5, w - wdx - 5, wdh,
           b ? b->Name() : "Error", 30, WIDGET_STR_CONST,
           NULL, this );

  wd = new ButtonWidget( B_ID_UNIT, 5, wd->TopEdge() + wdh + 5,
       wd->Width(), wdh, 0, "_Unit", this );
  wd->SetHook( this );

  e_unit = new StringWidget( 0, wdx, wd->TopEdge(),
           e_shop->Width(), wdh,
           mission.GetUnitSet().GetUnitInfo(event.GetData(1))->Name(),
           30, WIDGET_STR_CONST, NULL, this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventResearchWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdEventResearchWindow::WidgetActivated( Widget *widget, Window *win ) {
  TLWNode *n;

  switch( widget->ID() ) {
  case B_ID_UNIT:
    TLWListBuilder::BuildUnitTypesList( mission.GetUnitSet(), tlist );
    new ListSelectWindow( "Select researched unit", tlist,
        event.GetData(1), false, this, B_ID_UNIT_OK, view );
    break;
  case B_ID_UNIT_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 1, n ? n->ID() : -1 );
    e_unit->SetString( n ? n->Name() : "Error" );
    tlist.Clear();
    break;
  case B_ID_SHOP:
    TLWListBuilder::BuildShopList( mission.GetBuildings(), tlist );
    new ListSelectWindow( "Select factory", tlist,
        event.GetData(0), false, this, B_ID_SHOP_OK, view );
    break;
  case B_ID_SHOP_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 0, n ? n->ID() : -1 );
    e_shop->SetString( n ? n->Name() : "Error" );
    tlist.Clear();
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventScoreWindow::EdEventScoreWindow
// DESCRIPTION: This window allows configuration of a SCORE event.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdEventScoreWindow::EdEventScoreWindow( Event &event, Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 20 + 15, view->Width()),
           sfont->Height() * 4 + 50 );

  unsigned short wdx = sfont->Width() * 5 + 10;
  Widget *wd = new NumberWidget( N_ID_SCORE, wdx, 5,
               w - wdx - 5, sfont->Height() + 8,
               event.GetData(0), 1, 100,
               WIDGET_ALIGN_LEFT, "_Score", this );
  wd->SetHook( this );

  wd = new ButtonWidget( B_ID_TITLE, 5, wd->TopEdge() + wd->Height() + 5,
                    sfont->Width() * 10, wd->Height(), 0, "_Lose Title", this );
  wd->SetHook( this );

  wdx = wd->LeftEdge() + wd->Width() + 5;
  const char *str = mission.GetMessage(event.GetData(2));
  other_title = new StringWidget( 0, wdx, wd->TopEdge(), w - wdx - 5, wd->Height(),
           (str || event.GetData(2) == -1) ? str : "Error", 30, WIDGET_STR_CONST,
           NULL, this );

  wd = new ButtonWidget( B_ID_MSG, 5, wd->TopEdge() + wd->Height() + 5,
                         wd->Width(), wd->Height(), 0, "_Lose Msg", this );
  wd->SetHook( this );

  str = mission.GetMessage(event.GetData(1));
  other_msg = new StringWidget( 0, wdx, wd->TopEdge(), other_title->Width(), wd->Height(),
           (str || event.GetData(1) == -1) ? str : "Error", 30, WIDGET_STR_CONST,
           NULL, this );

  new ButtonWidget( GUI_CLOSE, 1, h - wd->Height() - 1, w - 2, wd->Height(),
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();

  TLWListBuilder::BuildMsgList( mission.GetMessages(), msglist );
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdEventScoreWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdEventScoreWindow::WidgetActivated( Widget *widget, Window *win ) {
  TLWNode *n;

  switch ( widget->ID() ) {
  case N_ID_SCORE:
    event.SetData( 0, static_cast<NumberWidget *>(widget)->Number() );
    break;
  case B_ID_MSG:
    new ListSelectWindow( "Select lose message", msglist, event.GetData(1),
                          true, this, B_ID_MSG_OK, view );
    break;
  case B_ID_MSG_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 1, n ? n->ID() : -1 );
    other_msg->SetString( n ? mission.GetMessage(n->ID()) : NULL );
    break;
  case B_ID_TITLE:
    new ListSelectWindow( "Select lose title", msglist, event.GetData(2),
                          true, this, B_ID_TITLE_OK, view );
    break;
  case B_ID_TITLE_OK:
    n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetData( 2, n ? n->ID() : -1 );
    other_title->SetString( n ? mission.GetMessage(n->ID()) : NULL );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigHaveShopWindow::EdTrigHaveShopWindow
// DESCRIPTION: This window allows configuration of a HAVE_BUILDING
//              trigger.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdTrigHaveShopWindow::EdTrigHaveShopWindow( Event &event,
                      Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 30 + 15, view->Width()),
           sfont->Height() * 4 + 50 );

  Widget *wd;
  unsigned short wdx, wdh = sfont->Height() + 8;

  wd = new ButtonWidget( B_ID_SHOP, 5, 5,
       sfont->Width() * 4 + 10, wdh, 0, "_Shop", this );
  wd->SetHook( this );

  Building *b = mission.GetBuildingByID( event.GetTData(0) );
  wdx = wd->LeftEdge() + wd->Width() + 5;
  t_shop = new StringWidget( 0, wdx, 5, w - wdx - 5, wdh,
           b ? b->Name() : "Error", 30, WIDGET_STR_CONST,
           NULL, this );

  wdx = sfont->Width() * 20 + 10;
  wd = new CycleWidget( B_ID_PLAYER, wdx, wd->Height() + 10,
       w - wdx - 5, wdh, 0, "Controlled by _Player",
       event.GetTData(1), player2_labels, this );
  wd->SetHook( this );

  wd = new NumberWidget( N_ID_TURN,
       sfont->Width() * 7 + 10, wd->TopEdge() + wd->Height() + 5,
       sfont->Width() * 6, wdh, event.GetTData(2),
       -1, 999, WIDGET_ALIGN_LEFT, "On _Turn", this );
  wd->SetHook( this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigHaveShopWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdTrigHaveShopWindow::WidgetActivated( Widget *widget, Window *win ) {

  switch( widget->ID() ) {
  case B_ID_SHOP:
    TLWListBuilder::BuildShopList( mission.GetBuildings(), tlist );
    new ListSelectWindow( "Select shop", tlist,
        event.GetTData(0), false, this, B_ID_SHOP_OK, view );
    break;
  case B_ID_SHOP_OK: {
    TLWNode *n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetTData( 0, n ? n->ID() : -1 );
    t_shop->SetString( n ? n->Name() : "Error" );
    tlist.Clear();
    break; }
  case B_ID_PLAYER:
    event.SetTData( 1, static_cast<CycleWidget *>(widget)->GetValue() );
    break;
  case N_ID_TURN:
    event.SetTData( 2, static_cast<NumberWidget *>(widget)->Number() );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigHaveUnitWindow::EdTrigHaveUnitWindow
// DESCRIPTION: This window allows configuration of a HAVE_UNIT trigger.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdTrigHaveUnitWindow::EdTrigHaveUnitWindow( Event &event,
                      Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 30 + 15, view->Width()),
           sfont->Height() * 4 + 50 );

  Widget *wd;
  unsigned short wdx, wdh = sfont->Height() + 8;
  char buf[32] = "Error";

  wd = new ButtonWidget( B_ID_UNIT, 5, 5,
       sfont->Width() * 4 + 10, wdh, 0, "_Unit", this );
  wd->SetHook( this );

  Unit *u = mission.GetUnitByID( event.GetTData(0) );
  if ( u ) sprintf( buf, "%s (%d)", u->Name(), u->ID() );

  wdx = wd->LeftEdge() + wd->Width() + 5;
  t_unit = new StringWidget( 0, wdx, 5, w - wdx - 5, wdh,
           buf, 30, WIDGET_STR_CONST, NULL, this );

  wdx = sfont->Width() * 20 + 10;
  wd = new CycleWidget( B_ID_PLAYER, wdx, wd->Height() + 10,
       w - wdx - 5, wdh, 0, "Controlled by _Player",
       event.GetTData(1), player2_labels, this );

  wd = new NumberWidget( N_ID_TURN,
       sfont->Width() * 7 + 10, wd->TopEdge() + wd->Height() + 5,
       sfont->Width() * 6, wdh, event.GetTData(2),
       -1, 999, WIDGET_ALIGN_LEFT, "On _Turn", this );
  wd->SetHook( this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );

  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigHaveUnitWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdTrigHaveUnitWindow::WidgetActivated( Widget *widget, Window *win ) {

  switch( widget->ID() ) {
  case B_ID_UNIT:
    TLWListBuilder::BuildUnitList( mission.GetUnits(), tlist );
    new ListSelectWindow( "Select unit", tlist,
        event.GetTData(0), false, this, B_ID_UNIT_OK, view );
    break;
  case B_ID_UNIT_OK: {
    TLWNode *n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetTData( 0, n ? n->ID() : -1 );
    t_unit->SetString( n ? n->Name() : "Error" );
    tlist.Clear();
    break; }
  case B_ID_PLAYER:
    event.SetTData( 1, static_cast<CycleWidget *>(widget)->GetValue() );
    break;
  case N_ID_TURN:
    event.SetTData( 2, static_cast<NumberWidget *>(widget)->Number() );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigTurnWindow::EdTrigTurnWindow
// DESCRIPTION: This window allows configuration of a TURN trigger.
// PARAMETERS : event   - event to edit
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdTrigTurnWindow::EdTrigTurnWindow( Event &event, View *view ) :
                  Window(WIN_CENTER, view), event(event) { 
  SetSize( MIN(sfont->Width() * 21 + 15, view->Width()),
           sfont->Height() * 2 + 30 );

  unsigned short wdx = sfont->Width() * 15 + 10,
                 wdh = sfont->Height() + 8;
  Widget *wd = new NumberWidget( 0, wdx, 5, w - wdx - 5, wdh,
       event.GetTData(0), 0, 999, WIDGET_ALIGN_LEFT,
       "Execute on _turn", this );
  wd->SetHook( this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigTurnWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdTrigTurnWindow::WidgetActivated( Widget *widget, Window *win ) {
  event.SetTData( 0, static_cast<NumberWidget *>(widget)->Number() );
  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigUnitDestroyedWindow::EdTrigUnitDestroyedWindow
// DESCRIPTION: This window allows configuration of a UNIT_DESTROYED
//              trigger.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdTrigUnitDestroyedWindow::EdTrigUnitDestroyedWindow( Event &event,
                           Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 30 + 15, view->Width()),
           sfont->Height() * 3 + 40 );

  Widget *wd;
  unsigned short wdx, wdh = sfont->Height() + 8;
  char buf[32] = "All";

  wd = new ButtonWidget( B_ID_UNIT, 5, 5,
       sfont->Width() * 4 + 10, wdh, 0, "_Unit", this );
  wd->SetHook( this );

  if ( event.GetTData(0) >= 0 ) {
    Unit *u = mission.GetUnitByID( event.GetTData(0) );
    if ( u ) sprintf( buf, "%s (%d)", u->Name(), u->ID() );
  }
  wdx = wd->LeftEdge() + wd->Width() + 5;
  t_unit = new StringWidget( 0, wdx, 5, w - wdx - 5, wdh,
           buf, 30, WIDGET_STR_CONST, NULL, this );

  wdx = sfont->Width() * 20 + 10;
  t_player = new CycleWidget( B_ID_PLAYER, wdx, wd->Height() + 10,
             w - wdx - 5, wdh,
             (event.GetTData(0) >= 0) ? WIDGET_DISABLED : 0,
             "Controlled by _Player", event.GetTData(1),
             player2_labels, this );
  t_player->SetHook( this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigUnitDestroyedWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdTrigUnitDestroyedWindow::WidgetActivated( Widget *widget, Window *win ) {

  switch( widget->ID() ) {
  case B_ID_UNIT:
    TLWListBuilder::BuildUnitList( mission.GetUnits(), tlist );
    new ListSelectWindow( "Select target unit", tlist,
        event.GetTData(0), true, this, B_ID_UNIT_OK, view );
    break;
  case B_ID_UNIT_OK: {
    TLWNode *n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetTData( 0, n ? n->ID() : -1 );
    t_unit->SetString( n ? n->Name() : "All", true );

    if ( n ) {
      event.SetTData( 1, static_cast<Unit *>(n->UserData())->Owner() );
      t_player->Disable();
    } else {
      event.SetTData( 1, event.Player()^1 );
      t_player->Enable();
      t_player->SetValue( event.GetTData(1) );
    }

    Draw();
    Show();
    tlist.Clear();
    break; }
  case B_ID_PLAYER:
    event.SetTData( 1, static_cast<CycleWidget *>(widget)->GetValue() );
  }

  return GUI_OK;
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigUnitPositionWindow::EdTrigUnitPositionWindow
// DESCRIPTION: This window allows configuration of a UNIT_POSITION
//              trigger.
// PARAMETERS : event   - event to edit
//              mission - current mission
//              view    - pointer to the window's view
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

EdTrigUnitPositionWindow::EdTrigUnitPositionWindow( Event &event,
                           Mission &mission, View *view ) :
      Window(WIN_CENTER, view), event(event), mission(mission) { 
  SetSize( MIN(sfont->Width() * 30 + 15, view->Width()),
           sfont->Height() * 3 + 40 );

  Widget *wd;
  unsigned short wdx, wdh = sfont->Height() + 8;
  char buf[32] = "Any";

  wd = new ButtonWidget( B_ID_UNIT, 5, 5,
       sfont->Width() * 4 + 10, wdh, 0, "_Unit", this );
  wd->SetHook( this );

  if ( event.GetTData(0) >= 0 ) {
    Unit *u = mission.GetUnitByID( event.GetTData(0) );
    if ( u ) sprintf( buf, "%s (%d)", u->Name(), u->ID() );
  }
  wdx = wd->LeftEdge() + wd->Width() + 5;
  t_unit = new StringWidget( 0, wdx, 5, w - wdx - 5, wdh,
           buf, 30, WIDGET_STR_CONST, NULL, this );

  wdx = sfont->Width() * 10 + 10;
  wd = new NumberWidget( N_ID_POSX,
       wdx, wd->TopEdge() + wd->Height() + 5,
       (w - wdx - sfont->Width() - 15) / 2, wdh,
       event.GetTData(1), 0, mission.GetMap().Width(),
       WIDGET_ALIGN_LEFT, "Position _X", this );
  wd->SetHook( this );

  wdx += wd->Width() + sfont->Width() + 10;
  wd = new NumberWidget( N_ID_POSY,
       wdx, wd->TopEdge(), w - wdx - 5, wdh,
       event.GetTData(2), 0, mission.GetMap().Height(),
       WIDGET_ALIGN_LEFT, "_Y", this );
  wd->SetHook( this );

  new ButtonWidget( GUI_CLOSE, 1, h - wdh - 1, w - 2, wdh,
                    WIDGET_DEFAULT, "_OK", this );
  Draw();
  Show();
}

////////////////////////////////////////////////////////////////////////
// NAME       : EdTrigUnitPositionWindow::WidgetActivated
// DESCRIPTION: React to user actions.
// PARAMETERS : widget - activated widget
//              win    - window containing the widget
// RETURNS    : GUI status
////////////////////////////////////////////////////////////////////////

GUI_Status EdTrigUnitPositionWindow::WidgetActivated( Widget *widget, Window *win ) {

  switch( widget->ID() ) {
  case B_ID_UNIT:
    TLWListBuilder::BuildUnitList( mission.GetUnits(), tlist );
    new ListSelectWindow( "Select unit", tlist,
        event.GetTData(0), true, this, B_ID_UNIT_OK, view );
    break;
  case B_ID_UNIT_OK: {
    TLWNode *n = static_cast<ListSelectWindow *>(win)->Selected();
    event.SetTData( 0, n ? n->ID() : -1 );
    t_unit->SetString( n ? n->Name() : "Any", true );
    tlist.Clear();
    break; }
  case N_ID_POSX:
    event.SetTData( 1, static_cast<NumberWidget *>(widget)->Number() );
    break;
  case N_ID_POSY:
    event.SetTData( 2, static_cast<NumberWidget *>(widget)->Number() );
  }

  return GUI_OK;
}

