--------------------------BEGIN-VHDL-LICENSE-----------------------------
-- fasu-test.vhdl - Testbench for Floating Point Add/Sub Execution Unit
-- Copyright (C) 2003-2004 -- SEMET Gaetan <gaetan@xeberon.net>
-- Copyright (C) 2000, 2001, 2003 Michael Riepe <michael@stud.uni-hannover.de>

-- This program is free software; you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2 of the License, or
-- (at your option) any later version.
--
-- This program is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.

-- You should have received a copy of the GNU General Public License
-- along with this program; if not, write to the Free Software
-- Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
---------------------------END-VHDL-LICENSE------------------------------

--pragma synthesis_off

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.numeric_std.all;
use std.textio.all;
use IEEE.std_logic_textio.all;

use work.FCPU_config.all;


entity fasu_test is
	generic (WIDTH : natural := 64);
end fasu_test;

architecture Arch_1 of fasu_test is

	component EU_FASU is
		generic (
			WIDTH : positive := 64
		);
	
		port(
		-- inputs : 
			-- operands
			Din_0    : in std_ulogic_vector(WIDTH-1 downto 0);
			Din_1    : in std_ulogic_vector(WIDTH-1 downto 0);
			-- subtract flag (should be derived from opcode)
			-- 0 for fadd, 1 for fsub
			Substract : in std_ulogic;
	    ieee_flag: in std_ulogic; -- 0 : ieee flag not set, 1: ieee flag set
			-- SIMD mode bits (not decoded)
			SIMD     : in std_ulogic_vector(1 downto 0);
			-- clock/reset/enable inputs
			Clk      : in std_ulogic;
			Rst      : in std_ulogic;
			En       : in std_ulogic;
	
		-- outputs:
			-- Result (32+32/64)
			Dout_0   : out std_ulogic_vector(WIDTH-1 downto 0);
			-- FPU Exception Flags
			FExout   : out std_ulogic_vector(8 downto 0)
			-- FExout(0) : DBZ  : Divide By Zero Detected
			-- FExout(1) : IOD  : Illegal Operand Detected
			-- FExout(2) : EUD  : Exponent Underflow Detected
			-- FExout(3) : EOD  : Exponent Overflow Detected
			-- FExout(4) : SNAN : Signaling Not A Number output
			-- FExout(5) : QNAN : Quiet Not A Number output
			-- FExout(6) : INF  : Infinite output
			-- FExout(7) : INE  : Inexact output
			-- FExout(8) : ZERO : Zero output 
		);
	end component;
	
	-- constants used in tests
	constant SGL_SIZE           : natural := 32;
	constant DBL_SIZE           : natural := 64;
	
	-- numbers:
	constant const32_0d3568     :std_ulogic_vector(SGL_SIZE-1 downto 0)
	                            := "00111110101101101010111001111101"; --  0.3568
	constant const32_0d3569     :std_ulogic_vector(SGL_SIZE-1 downto 0)  
		                          := "00111110101101101011101110011001"; --  0.3569
	constant const32_0d35692215 :std_ulogic_vector(SGL_SIZE-1 downto 0)  
		                          := "00111110101101101011111010000000";
		                                                            -- ~ 0.35692215
	constant const32_0d35692212 :std_ulogic_vector(SGL_SIZE-1 downto 0)  
		                          := "00111110101101101011111001111111";
		                                                            -- ~ 0.35692212
	constant const32_0          :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "00000000000000000000000000000000"; --  0
	constant const32_m_0        :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "10000000000000000000000000000000"; -- -0
	constant const32_PI         :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000000010010010000111111010000"; -- 3.14159
	constant const32_6          :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000000110000000000000000000000"; --  6
	constant const32_m6         :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "11000000110000000000000000000000"; -- -6
	constant const32_9          :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000001000100000000000000000000"; --  9
	constant const32_m9         :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "11000001000100000000000000000000"; --  9
	constant const32_9d58448    :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000001000110010101101000001000"; -- 9.58448
	constant const32_15         :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000001011100000000000000000000"; --  15
	constant const32_m15        :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "11000001011100000000000000000000"; -- -15
	constant const32_15d58448   :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000001011110010101101000001000";-- 15.58448
	constant const32_18d7868    :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000001100101100100101101011110";-- 18.7868
	constant const32_21         :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000001101010000000000000000000"; --  21
	constant const32_m21        :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "11000001101010000000000000000000"; --  -21
	constant const32_28d37128   :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000001111000101111100001100010"; --28.37128
	constant const32_m994       :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "11000100011110001000000000000000"; --   -994
	constant const32_1000       :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000100011110100000000000000000"; --   1000
	constant const32_m1000      :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "11000100011110100000000000000000"; --  -1000
	constant const32_1006       :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01000100011110111000000000000000"; --   1006
	constant const32_1e21       :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "01100010010110001101011100100111"; --  10^21
	constant const32_1em21      :std_ulogic_vector(SGL_SIZE-1 downto 0)    
		                          := "00011100100101110001110110100000"; --  10^-21
	
	-- special constants:
  constant const32_zero    : std_ulogic_vector   
		                       := "00000000000000000000000000000000";
  constant const32_m_zero  : std_ulogic_vector    
		                       := "10000000000000000000000000000000";
  constant const32_infty   : std_ulogic_vector    
		                       := "01111111100000000000000000000000";
  constant const32_m_infty : std_ulogic_vector    
		                       := "11111111100000000000000000000000";
  constant const32_SNaN    : std_ulogic_vector    
		                       := "01111111100000000000000000000001"; --    SNaN
  constant const32_QNaN    : std_ulogic_vector    
		                       := "01111111110000000000000000000001"; --    QNaN
  
  constant CYCLEDURATION : time := 10 ns;
  constant CYCLEDURATION_DIV2 : time := 5 ns;

	signal Din_0, Din_1 : F_VECTOR;
	signal Substract, Rst, En : std_ulogic;
	signal Clk : std_ulogic := '0';
	signal Flags : std_ulogic_vector(23 downto 18);
	signal SIMD : std_ulogic_vector(1 downto 0);
	signal ieee_flag : std_ulogic;
	signal Dout_0 : std_ulogic_vector(WIDTH-1 downto 0);
	signal FExout : std_ulogic_vector(8 downto 0);

	-- CONVERSION PROCEDURES
	
	
	-- TODO: conversion routines from REAL to std_ulogic_vector
	-- but no pointer in VHDL...
	
	procedure print_single_vector (lbl : string;
							x : std_ulogic_vector;
							sf: string;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des);
		write(lout, x);
		write(lout," (");
		write(lout,sf);
		write(lout,")");
		writeline(output, lout);
	end print_single_vector;

	procedure print_double_vector (lbl : string;
							x : std_ulogic_vector;
							sf: string;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des);
		write(lout, x);
		write(lout," (");
		write(lout,sf);
		write(lout,")");
		writeline(output, lout);
	end print_double_vector;


	-- REPORT PROCEDURES

	procedure print_str (s : string) is
		variable lout : line;
	begin
		write(lout, s);
		writeline(output, lout);
	end print_str;
	
	procedure print_stdval (lbl : string;
							x : std_ulogic;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_stdval;

	procedure print_vector (lbl : string;
							x : std_ulogic_vector;
							des : string := " := ") is
		variable lout : line;
	begin
		write(lout, lbl & des); write(lout, x); writeline(output, lout);
	end print_vector;

	procedure do_error (lbl : string;
						a_x, a_y : std_ulogic_vector) is
	begin
		print_str("WHOA THERE!!!");
--		print_signals;
		print_vector(lbl, a_x);
		print_vector(lbl, a_y, " /= ");
	end do_error;

	procedure check_numeric (lbl : string;
							 x : std_ulogic_vector;
							 y : natural) is
		variable tmp : std_ulogic_vector(x'range);
		variable lout : line;
	begin
		tmp := std_ulogic_vector(to_unsigned(y mod 2**x'length, x'length));
		if x /= tmp then
			do_error(lbl, x, tmp);
		end if;
	end check_numeric;

  -- TODO : find a way to have y as REAL and not as std_ulogic_vector
	procedure check_float (lbl : string;
							 x : std_ulogic_vector;
							 y : std_ulogic_vector;
							 sf: string) is
		variable lout : line;
		constant chk: string := "Check successfull: ";
	begin
		if x /= y then
			do_error(lbl, x, y);
		else
		  write(lout,chk & lbl);
		  write(lout, " := ");
		  write(lout,x);
		  write(lout," (");
		  write(lout,sf);
		  write(lout,")");
		  writeline(output, lout);
		end if;
	end check_float;
	
	procedure print_signals is
	begin
	  print_str   ("** SIGNALS REPORT **");
	  print_str   ("Inputs:");
		print_stdval("En       ", En);
		print_stdval("Rst      ", Rst);
		print_vector("Din_0    ", Din_0);
		print_vector("Din_1    ", Din_1);
		print_stdval("Substract", Substract);
		print_vector("Flags    ", Flags);
		print_vector("SIMD     ", SIMD);
		print_str   ("Outputs:");
		print_vector("Dout_0   ", Dout_0);
	  print_vector("FExout   ", FExout);
	  print_str   ("** END OF SIGNALS REPORT **");	
	end;

-- TEST PROCESSES AND COMPONENTS
begin

  -- COMPONENT TO TEST
  
	testunit32 : EU_FASU 
		generic map ( WIDTH => 64)
		port map (Din_0 => Din_0,
							Din_1 => Din_1,
							ieee_flag => ieee_flag,
							Substract => Substract,
							SIMD => SIMD,
							Clk => Clk,
							Rst => Rst,
							En => En,
							Dout_0 => Dout_0,
							FExout => FExout);


  -- Clock generator process
  ClkProcess: process(clk)
  begin
	  Clk <= not clk after CYCLEDURATION_DIV2;
	end process;
	
	-- driver process
	testproc : process
	variable lout : line;
	begin
		print_str("*** testing FPU add/sub unit (fadd/fsub instruction) ***");
		print_str("FASU size : 64 bit (2x32-bit SIMD)");
		
		write(lout, "Clock period: ");
		write(lout, CYCLEDURATION);
		writeline(output, lout);
		
		Din_0 <= (others => '0');
		Din_1 <= (others => '0');
		Substract <= '0';
		Flags <= (others => '0');
		SIMD <= (others => '0');
		En <= '0';
		Rst <= '0';
		ieee_flag <= '0';
		
	
		print_str("Reseting unit...");
		Rst <= '1';
		
		wait for CYCLEDURATION/2 + CYCLEDURATION - CYCLEDURATION/5;
		
		Rst <= '0';
		print_str("End of reset.");
		
		wait for 15 ns;
		
		print_str("Default values:");
		print_signals;
		print_str("------ FLOAT ADDITION (SIMD=00) -----");
		print_str("First calculation: single float");
		print_single_vector("Din_0",const32_15, "15.0f");
		print_single_vector("Din_1",const32_6,"6.0f");
		Din_0(SGL_SIZE-1 downto 0)     <= const32_1000(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= const32_1000(SGL_SIZE-1 downto 0);
		Din_1(SGL_SIZE-1 downto 0)     <= const32_6(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= const32_6(SGL_SIZE-1 downto 0);
		En <= '1';		
		print_str("Launching ONE calculation (over 6 cycles)...");
		
		wait for 6*CYCLEDURATION;
		
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0), const32_21, "21.0f");
		
		print_str("Using SIMD: +inf on Din_0(high) + -15 on Din_1(hight bits)...");
		Din_0(WIDTH-1 downto SGL_SIZE) <=  const32_infty;
		Din_1(WIDTH-1 downto SGL_SIZE) <=  const32_15;
		print_str("Launching ONE calculation (over 6 cycles)...");
		
		wait for 6*CYCLEDURATION;
		
	  check_float("Dout_0", Dout_0(WIDTH-1 downto SGL_SIZE), 
	              const32_infty, "+inf");
	
		En <= '0';
		
		wait for 2*CYCLEDURATION;
		
		
		En <= '1';
		
		print_str("Launching cascading calculations (6 calculations)");
		print_str("1st cycle");
		print_single_vector("Din_0", const32_6, "6.0f");
		print_single_vector("Din_1", const32_m15, "-15.0f");
  	Din_0(SGL_SIZE-1 downto 0) <=  const32_6(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		Din_1(SGL_SIZE-1 downto 0) <=  const32_m15(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= (others => '0');		
		
		wait for CYCLEDURATION;
		
		print_str("2nd cycle");
		print_single_vector("Din_0", const32_m6, "-6.0f");
		print_single_vector("Din_1", const32_m15, "-15.0f");
  	Din_0(SGL_SIZE-1 downto 0) <=  const32_m6(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		Din_1(SGL_SIZE-1 downto 0) <=  const32_m15(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= (others => '0');		
		
		wait for CYCLEDURATION;
		
		print_str("3rd cycle");
		print_single_vector("Din_0", const32_m6, "-6.0f");
		print_single_vector("Din_1", const32_15d58448, "15.58448f");
  	Din_0(SGL_SIZE-1 downto 0) <=  const32_m6(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		Din_1(SGL_SIZE-1 downto 0) <=  const32_15d58448(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		
		wait for CYCLEDURATION;
		
		print_str("4th cycle");
		print_single_vector("Din_0", const32_6, "6.0f");
		print_single_vector("Din_1", const32_1000, "1000.0f");
  	Din_0(SGL_SIZE-1 downto 0) <=  const32_6(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		Din_1(SGL_SIZE-1 downto 0) <=  const32_1000(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		
		wait for CYCLEDURATION;
		
		print_str("5th cycle");
		print_single_vector("Din_0", const32_6, "6.0f");
		print_single_vector("Din_1", const32_1000, "-1000.0f");
  	Din_0(SGL_SIZE-1 downto 0) <=  const32_6(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		Din_1(SGL_SIZE-1 downto 0) <=  const32_m1000(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		
		wait for CYCLEDURATION;
		
		print_str("6th cycle");
		print_single_vector("Din_0", const32_1000, "1000.0f");
		print_single_vector("Din_1", const32_1e21, "1e21f");
		Din_0(SGL_SIZE-1 downto 0) <=  const32_1000(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		Din_1(SGL_SIZE-1 downto 0) <=  const32_1e21(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		
		print_str("Result first calculation (6.0f + -15.0f)");
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0), const32_m9, "-9.0f");
		wait for CYCLEDURATION;
		
		print_str("7th cycle");
		print_single_vector("Din_0", const32_9d58448, "9.58448f");
		print_single_vector("Din_1", const32_18d7868, "18.7848f");
		Din_0(SGL_SIZE-1 downto 0) <=  const32_9d58448(SGL_SIZE-1 downto 0);
		Din_0(WIDTH-1 downto SGL_SIZE) <= (others => '0');
		Din_1(SGL_SIZE-1 downto 0) <=  const32_18d7868(SGL_SIZE-1 downto 0);
		Din_1(WIDTH-1 downto SGL_SIZE) <= (others => '0');

		print_str("Result second calculation (-6.0f + -15.0f)");
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0), const32_m21, "-21.0f");
		
		wait for CYCLEDURATION;
		
		print_str("8th cycle");
		print_str("Result third calculation (-6.0f + 15.58448f)");
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0), 
		            const32_9d58448, "9.58448f");
		
		wait for CYCLEDURATION;
		
		print_str("9th cycle");
		print_str("Result fourth calculation (6.0f + 1000.0f)");
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0), 
		            const32_1006, "1006.0f");
		
		wait for CYCLEDURATION;
		
		print_str("10th cycle");
		print_str("Result fifth calculation (6.0f + -1000.0f)");
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0),
		            const32_m994, "-994.0f");
		
		wait for CYCLEDURATION;
		
		print_str("11th cycle");
		print_str("Result sixth calculation (1000.0f + 1e21f)");
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0),
		            const32_1e21, "1e21f");
		
		wait for CYCLEDURATION;
		
		print_str("12th cycle");
		print_str("Result seventh calculation (9.58448f + 18.7868)");
		check_float("Dout_0", Dout_0(SGL_SIZE-1 downto 0),
		            const32_28d37128, "28,37128");
		            
		wait for 2*CYCLEDURATION;
		print_str("Start new tests : 32 bit SIMD mode");
		print_str("            Din_0    Din_1");
		print_str("high SIMD : +infty + 0");
		print_str("low  SIMD : -infty + +infty");
		
		Din_0(WIDTH-1 downto SGL_SIZE) <= const32_infty;
		Din_1(WIDTH-1 downto SGL_SIZE) <= const32_zero;
		Din_0(SGL_SIZE-1 downto 0) <=  const32_m_infty;
		Din_1(SGL_SIZE-1 downto 0) <=  const32_infty;
		print_double_vector("Din_0",(const32_infty & const32_m_infty), "(+infty, -infty)");
		print_double_vector("Din_1", (const32_zero & const32_infty), "( 0, +infty)");
		wait;
	end process;

end Arch_1 ;

-- simili specific:
-- recreate waveform:
-- source "Waveform0-wave.tcl"

-- custom convert program command:
-- convert from hexadecimal binary representation to float:
-- floatrepres.exe -h 4c00000

-- convert from binary representation to float:
-- floatrepres.exe -b 100101100010...

-- convert from float to bin:
-- floatrepres.exe -f 10.5



















