#include <PenguinPlay/PenguinPlay.h>
#include <PenguinPlay/Codec.h>

static const int adpcmStepTable[89] =
{
  7,     8,     9,    10,    11,    12,    13,    14,
  16,    17,    19,    21,    23,    25,    28,    31,
  34,    37,    41,    45,    50,    55,    60,    66,
  73,    80,    88,    97,   107,   118,   130,   143,
  157,   173,   190,   209,   230,   253,   279,   307,
  337,   371,   408,   449,   494,   544,   598,   658,
  724,   796,   876,   963,  1060,  1166,  1282,  1411,
  1552,  1707,  1878,  2066,  2272,  2499,  2749,  3024,
  3327,  3660,  4026,  4428,  4871,  5358,  5894,  6484,
  7132,  7845,  8630,  9493, 10442, 11487, 12635, 13899,
  15289, 16818, 18500, 20350, 22385, 24623, 27086, 29794,
  32767
};

static const int adpcmIndexTable[16] =
{
  -1, -1, -1, -1, 2, 4, 6, 8, -1, -1, -1, -1, 2, 4, 6, 8
};

static const signed char fibonacci[16] =
{
  -34, -21, -13, -8, -5, -3, -2, -1, 0, 1, 2, 3, 5, 8, 13, 21
};

static const signed char exponential[16] =
{
  -128, -64, -32, -16, -8, -4, -2, -1, 0, 1, 2, 4, 8, 16, 32, 64
};

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN


inline unsigned long Codec::Convert32(unsigned long x)
{
  return (((x & 0xff000000) >> 24) |
          ((x & 0x00ff0000) >>  8) |
          ((x & 0x0000ff00) <<  8) |
          ((x & 0x000000ff) << 24));
}

// assuming that unsigned short is 16 bits
inline unsigned short Codec::Convert16(unsigned short x)
{
  return (((x & 0xff00) >> 8) |
          ((x & 0x00ff) << 8));
}

#ifdef PP_WORDS_BIGENDIAN
  unsigned short Codec::LE16(unsigned short x) {return Convert16(x);}
  unsigned long  Codec::LE32(unsigned long  x) {return Convert32(x);}
  unsigned short Codec::BE16(unsigned short x) {return x;}
  unsigned long  Codec::BE32(unsigned long  x) {return x;}
#else
  unsigned short Codec::LE16(unsigned short x) {return x;}
  unsigned long  Codec::LE32(unsigned long  x) {return x;}
  unsigned short Codec::BE16(unsigned short x) {return Convert16(x);}
  unsigned long  Codec::BE32(unsigned long  x) {return Convert32(x);}
#endif




unsigned short Codec::DecodeMuLawSample(unsigned char ulaw)
{
  ulaw = ~ulaw;
  const unsigned char exponent = (ulaw >> 4)   & 0x07;
  const unsigned char mantissa = (ulaw & 0x0f) + 16;
  unsigned long adjusted = (mantissa << (exponent + 3)) - 128 - 4;
  return (ulaw & 0x80) ? adjusted : -adjusted;
}

bool Codec::DecodeMuLaw(const unsigned char*   src,
                           unsigned short*& dest,
                           unsigned long    length)
{
  static unsigned short* muLawDecodeTable = 0;
  if (muLawDecodeTable == 0)
    {
      muLawDecodeTable = new unsigned short [256];
      if (muLawDecodeTable == 0)
        return false; // throw something ?
      for (int i = 0; i < 256; ++i)
        muLawDecodeTable[i] = DecodeMuLawSample(i);
    }

  for (unsigned long i = 0; i < length; ++i)
    dest[i] = muLawDecodeTable[src[i]];

  return true;
}

unsigned short Codec::DecodeALawSample(unsigned char alaw)
{
  alaw ^=0x55;
  const unsigned char exponent = (alaw >> 4)  & 0x07;
  const unsigned char mantissa = (alaw & 0xf) + (exponent ? 16 : 0);
  unsigned long adjusted = (mantissa << (exponent + 4));
  return (alaw & 0x80) ? -adjusted : adjusted;
}

bool Codec::DecodeALaw(const unsigned char* src,
                          unsigned short*&     dest,
                          unsigned long        length)
{
  static unsigned short* aLawDecodeTable = 0;
  if (aLawDecodeTable == 0)
    {
      aLawDecodeTable = new unsigned short [256];
      if (aLawDecodeTable == 0)
        return false;
      for (int i = 0; i < 256; ++i)
        aLawDecodeTable[i] = DecodeALawSample(i)<<8;
    }

  for (unsigned long i = 0; i < length; ++i)
    dest[i] = aLawDecodeTable[src[i]];

  return true;
}

bool Codec::DecodeFibonacciDPCM(const unsigned char* src,
                                   unsigned char*&      dest,
                                   unsigned long        length)
{
  signed char previous = src[0];
  dest[0] = previous ^ 0x80;

  for (unsigned long i = 0; i < length-1; ++i)
    {
      previous += fibonacci[(src[i+1]>>4) & 0x0f];
      (signed char&)dest[i*2] = previous ^ 0x80;
      previous += fibonacci[src[i+1] & 0x0f];
      (signed char&)dest[i*2 + 1] = previous ^ 0x80;
    }

  return true;
}

bool Codec::DecodeExponentialDPCM(const unsigned char* src,
                                     unsigned char*&      dest,
                                     unsigned long        length)
{
  signed char previous = src[0];
  dest[0] = previous^0x80;
  for (unsigned long i = 0; i < length-1; ++i)
    {
      previous += exponential[(src[i+1]>>4)&0xf];
      dest[i*2] = previous^0x80;
      previous += exponential[src[i+1]&0xf];
      dest[i*2+1] = previous^0x80;
    }
  return true;
}


//============================================================================
// DecodeADPCM
//----------------------------------------------------------------------------
/** Decode Adaptive Differential Pulse Code Modulation

   Compressed Samples are stored in four bits. Take the value of the stored
   sample, multiply it by the current step size and add it to the previous
   sample and you have the uncompressed value of the sample.

   The delta is the difference between the last sample and the current one.

   The step size is not stored directly instead 88 possible values are stored
   in a table.
*/
//============================================================================
long Codec::DecodeAdpcmSample(unsigned char adpcm,
                                 long&         previous,
                                 int           step_table_index)
{
  long decoded_value;

  int  delta;

  // Calculate the difference
  // this is difference = (vpdiff +.5)*step/4
  delta = adpcmStepTable[step_table_index] >> 3;
  if (adpcm & 1) delta += adpcmStepTable[step_table_index] >> 2;
  if (adpcm & 2) delta += adpcmStepTable[step_table_index] >> 1;
  if (adpcm & 4) delta += adpcmStepTable[step_table_index];

  // Now add the delta we just calculated to
  // the previous value.

  // Check the sign bit
  if (adpcm & 8) decoded_value = previous - delta;
  else           decoded_value = previous + delta;

  // Check for overflow and underflow errors
  if (decoded_value > 32767)        decoded_value = 32767;
  else if (decoded_value < -32768)  decoded_value = -32768;

  // copy the value to the previous value so it can be used in the next call.
  previous = decoded_value;

  // Adjust the Step Table Index
  step_table_index += adpcmIndexTable[adpcm];

  // Check for step table index for
  // overflow and underflow (step table is 88 elements long)
  if (step_table_index > 88)     step_table_index = 88;
  else if (step_table_index < 0) step_table_index = 0;

  return decoded_value;
}


unsigned char Codec::EncodeAdpcmSample(unsigned char sample,
                                       long&         previous,
                                       int&          step_table_index)
{
  int diff  = sample - previous;
  int step  = adpcmStepTable[step_table_index];
  int adpcm = 0;

  // set sign bit
  if (diff < 0) { adpcm = 8; diff = -diff; }

  // this is essentially delta = (diff<<2)/step
  // except the roundoff is handled differently
  if (diff >= step)
    {
      adpcm |= 4 ;
      diff -= step;
    }
  step >>=1;
  if (diff >= step)
    {
      adpcm |= 2;
      diff -= step;
    }
  step >>=1;
  if (diff >=step)
    {
      adpcm |= 1;
      diff -= step;
    }

  // update the state
  DecodeAdpcmSample(adpcm, previous, step_table_index);
  return adpcm;
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END

