//=============================================================================
// $Id: DspMmapped.cpp,v 1.3 1999/11/29 17:12:57 creinig Exp $
//-----------------------------------------------------------------------------
// $Log: DspMmapped.cpp,v $
// Revision 1.3  1999/11/29 17:12:57  creinig
// fixed little signedness issue in munmap parameter passing
//
// Revision 1.2  1999/11/28 23:47:25  creinig
// adapted sources to use ppconfig(_win32).h
//
// Revision 1.1.1.1  1999/11/06 11:54:00  creinig
// Back in CVS at last
//
//=============================================================================

#include <PenguinPlay/PenguinPlay.h>
#include <PenguinPlay/DspMmapped.h>
#include <PenguinPlay/Sample.h>
#include <PenguinPlay/SoftwareMixer.h>

//#ifdef PP_DEBUG
//#include <stdio.h>
//#endif

#include <sys/ioctl.h>
#include <sys/soundcard.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <fcntl.h>
#include <unistd.h> // sleep, close, access

#include <assert.h>

#include <sys/mman.h>

#include <string>

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

//=============================================================================

DspMmapped::DspMmapped(int                rate,
                       int                bits,
                       bool               stereo,
                       const std::string& device)
     : m_device(device)
{
  Create(rate, bits, stereo ? 1 : 0);
}

//=============================================================================

DspMmapped::~DspMmapped()
{
  try
    {
      Destroy();
    }
  catch(...)
    {
    }
}

void DspMmapped::Create(int rate, int bits, int stereo)
{
  m_rate    = rate;
  m_bits    = bits;

  if (m_fd == -1)
    {
      // dont open device if it is already open
      if (access(m_device.c_str(), W_OK) != 0)
        {
	  ppWarning ("Mmapped %s: Can't access.", m_device.c_str ());
          ppThrow (EAccessFailure, "Can't access.");
        }
      // close(0) // -- ?
      if ((m_fd = open(m_device.c_str(), O_RDWR, 0)) == -1)
	{
          ppWarning ("Mmapped %s: can't open for output.", m_device.c_str ());
          ppThrow (EAccessFailure, "can't open for output.");
	}
    }

  // is there stereo?
  if (ioctl(m_fd, SNDCTL_DSP_STEREO, &stereo) == -1)
    {
      ppWarning ("Mmapped %s: Error setting stereo.", m_device.c_str ());
      ppThrow (EUnavailable, "Error setting stereo.");
    }

  m_stereo = (stereo != 0);

  // can I set the dsp to the rate given?
  if (ioctl(m_fd, SNDCTL_DSP_SPEED, &m_rate) == -1)
    {
      ppWarning ("Mmapped %s: Error setting rate.", m_device.c_str ());
      ppThrow (EUnavailable, "Error setting rate.");
    }

  // Can the device play 16bit samples?
  if (ioctl(m_fd,SNDCTL_DSP_SAMPLESIZE,&m_bits) == -1)
    {
         ppWarning ("Mmapped %s: Error setting sample size.", m_device.c_str ());
         ppThrow (EUnavailable, "Error setting sample size.");
    }

  if (m_stereo)
    ppDebug1 ("Mmapped %s : %d bit %d Hz stereo", m_device.c_str(),m_bits,m_rate);
  else
    ppDebug1 ("Mmapped %s : %d bit %d Hz mono", m_device.c_str(),m_bits,m_rate);

  int caps;
  if (ioctl(m_fd, SNDCTL_DSP_GETCAPS, &caps)==-1)
    {
      ppWarning ("Can't get capabilities for %s", m_device.c_str ());
      ppThrow (EHardwareUnavailable, "Can't get capabilities");
    }
  // the device must support TRIGGER and MMAP
  if (!(caps & DSP_CAP_TRIGGER) ||
      !(caps & DSP_CAP_MMAP))
    {
      ppWarning ("%s doesn't support mmapping", m_device.c_str ());
      ppThrow (EUnavailable, "mmapping unsupported by device");
    }

  int frag = 0xffff000c;	/* Max # fragments of 2^13=8k bytes */
  ioctl(m_fd, SNDCTL_DSP_SETFRAGMENT, &frag);

  audio_buf_info info;
  if (ioctl(m_fd, SNDCTL_DSP_GETOSPACE, &info)==-1)
    {
      ppWarning ("Mmapped %s: error querying SNDCTL_DSP_GETOPSPACE", m_device.c_str ());
      ppThrow (EAccessFailure, "error querying SNDCTL_DSP_GETOPSPACE");
    }

  m_fragsize = info.fragsize;
  m_size     = info.fragsize * info.fragstotal;

  ppDebug1 ("Mmapped %s : size %d\n", m_device.c_str(), m_size);

  m_buffer_size = m_fragsize;

  if (m_stereo == 1)
    m_buffer_size >>= 1;

  if (m_bits == 16)
    m_buffer_size >>= 1;

  // mmap is supposed to return a caddr_t but it doesn't seem to
  m_audio_buffer=(unsigned char*)mmap(NULL,
                                      m_size,
                                      PROT_WRITE,
                                      MAP_SHARED,
                                      m_fd,
                                      0);
  if (m_audio_buffer == MAP_FAILED)
    {
      m_audio_buffer = 0;
      ppWarning ("Error mmapping %s", m_device.c_str ());
      ppThrow (EAccessFailure, "Error mmapping devide");
    }


  int tmp = 0;
  ioctl(m_fd, SNDCTL_DSP_SETTRIGGER, &tmp);
  tmp = PCM_ENABLE_OUTPUT;
  ioctl(m_fd, SNDCTL_DSP_SETTRIGGER, &tmp);
}

//=============================================================================

void DspMmapped::Destroy()
{
  // stop making noises
  int tmp = 0;
  ioctl(m_fd, SNDCTL_DSP_SETTRIGGER, &tmp);
  // unmap the buffer
  if (munmap((char *) m_audio_buffer, m_size) == -1)
    ppWarning ("Error unmapping memory mapped %s", m_device.c_str());
  m_audio_buffer = 0;
  close(m_fd);
  m_fd = -1;
}


//=============================================================================
// Mixer Functions
//=============================================================================


//=============================================================================
///
inline void DspMmapped::MixBuffer(Status&       status,
                                     unsigned long length,
                                     Fixed      scale)
{
   unsigned long left;
   unsigned long right;

   status.CalculateVolume(left, right);

   using SoftwareMixer::MixStereoUnrolled;

   // mix the sample into the mix_buffer
   if (status.m_sample->Is16Bit())
      {
         if (m_bits == 16)
            {
               MixStereoUnrolled<int16_t, int16_t, 10>
                  (length,
                   left,
                   right,
                   scale,
                   status.m_position,
                   (int16_t*)status.m_sample->GetData(),
                   (int16_t*)m_audio_buffer_ptr);
            }
         else
            {
               MixStereoUnrolled<int16_t, u_int8_t, 10>
                  (length,
                   left,
                   right,
                   scale,
                   status.m_position,
                   (int16_t*)status.m_sample->GetData(),
                   (u_int8_t*)m_audio_buffer_ptr);
            }
      }
   else
      {
         if (m_bits == 16)
            {
               MixStereoUnrolled<u_int8_t, int16_t, 10>
                  (length,
                   left,
                   right,
                   scale,
                   status.m_position,
                   (u_int8_t*)status.m_sample->GetData(),
                   (int16_t*)m_audio_buffer_ptr);
            }
         else
            {
               MixStereoUnrolled<u_int8_t, u_int8_t, 4>
                  (length,
                   left,
                   right,
                   scale,
                   status.m_position,
                   (u_int8_t*)status.m_sample->GetData(),
                   (u_int8_t*)m_audio_buffer_ptr);

            }
      }
}

//=============================================================================
///
void DspMmapped::PrepareBuffer()
{
  fd_set writeset;

  FD_ZERO(&writeset);
  FD_SET(m_fd, &writeset);

  timeval tim;

  tim.tv_sec = 10;
  tim.tv_usec= 0;
  select(m_fd+1, &writeset, &writeset, NULL, NULL);

  struct count_info count;

  if (ioctl(m_fd, SNDCTL_DSP_GETOPTR, &count)==-1)
    {
      ppWarning ("Mmapped %s : error querying GETOPTR", m_device.c_str ());
      ppThrow (EUnavailable, "error querying GETOPTR");
    }
  // align the pointer to fragment boundaries
  count.ptr = (count.ptr/m_fragsize)*m_fragsize;

  m_audio_buffer_ptr = m_audio_buffer+count.ptr;

  // set a few bytes in the next fragment

  if ((count.ptr+m_fragsize+16) < m_size)
    m_extra = 16;
  else
    m_extra = 0;

  m_mix_length = m_buffer_size;//+m_extra;

  // first clear the fragment
  memset(m_audio_buffer_ptr, 0, m_fragsize/*+m_extra*/);
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END
