#include <PenguinPlay/PenguinPlay.h>

#include <fstream>
#include <string>
#include <stdio.h>
#include <string.h>
#include <sys/stat.h>
#include <PenguinPlay/SampleWav.h>
#include <PenguinPlay/Codec.h>

PP_NAMESPACE_BEGIN
PP_I_NAMESPACE_BEGIN

class SampleWav::Impl
{
public:
  Impl(SampleWav& sample);
  ~Impl();

  struct RiffHeader;
  struct FmtChunk;

  bool GetRiffWaveHeader(FILE* fp, RiffHeader& riff);
  bool ReadFmtChunk(FILE* fp,  FmtChunk& fmt,        u_int32_t chunksize);
  bool ReadDataChunk(FILE* fp, const FmtChunk& fmt,  u_int32_t chunksize);
  bool ReadInfoChunk(FILE* fp, u_int32_t chunkid, u_int32_t chunksize);
  bool ReadListChunk(FILE* fp, u_int32_t chunksize);

  bool ReadChunk(FILE*      fp,
                 u_int32_t  chunkid,
                 u_int32_t& chunksize,
                 FmtChunk*& fmt);

  bool ReadInChunks(FILE* fp, RiffHeader& riff);

  struct CuePoints;
  struct PlayList;
  struct ADPCMCOEFSET;
  struct ADPCMBLOCKHEADER;
  struct DVI_ADPCMBLOCKHEADER;

  SampleWav&    m_sample;
  u_int32_t     m_fact_file_size;
  char*         m_comment;
  char*         m_copyright;
  char*         m_creation_date;
  char*         m_engineer;
  char*         m_genre;
  char*         m_keywords;
  char*         m_medium;
  char*         m_product;
  char*         m_subject;
  char*         m_software;
  char*         m_source;
  char*         m_src_reference;
  char*         m_technician;
  ADPCMCOEFSET* m_coefset;
};

//=============================================================================
//     THE RIFF WAVE FORMAT
//=============================================================================

//============================================================================
/// WAVE FORMAT TYPES
//============================================================================
enum WAVE
{
  WAVE_FORMAT_UNKNOWN               =(0x0000),
  WAVE_FORMAT_PCM                   =(0x0001),
  WAVE_FORMAT_ADPCM                 =(0x0002),
  WAVE_FORMAT_IBM_CVSD              =(0x0005),
  WAVE_FORMAT_ALAW                  =(0x0006),
  WAVE_FORMAT_MULAW                 =(0x0007),
  WAVE_FORMAT_OKI_ADPCM             =(0x0010),
  WAVE_FORMAT_DVI_ADPCM             =(0x0011),
  WAVE_FORMAT_IMA_ADPCM             =(WAVE_FORMAT_DVI_ADPCM),
  WAVE_FORMAT_DIGISTD               =(0x0015),
  WAVE_FORMAT_DIGIFIX               =(0x0016),
  WAVE_FORMAT_YAMAHA_ADPCM          =(0x0020),
  WAVE_FORMAT_SONARC                =(0x0021),
  WAVE_FORMAT_DSPGROUP_TRUESPEECH   =(0x0022),
  WAVE_FORMAT_ECHOSC1               =(0x0023),
  WAVE_FORAMT_AUDIOFILE_AF18        =(0x0024),
  WAVE_FORMAT_IBM_MULAW             =(0x0101),
  WAVE_FORMAT_IBM_ALAW              =(0x0102),
  WAVE_FORMAT_IBM_ADPCM             =(0x0103),
  WAVE_FORMAT_CREATIVE_ADPCM        =(0x0200)
};

//============================================================================
/// CHUNK ID's
//============================================================================
enum CKID_RIFF
{
  CKID_RIFF_FMT  =(0x20746d66), ///< "fmt "
  CKID_RIFF_DATA =(0x61746164), ///< "data"
  CKID_RIFF_FACT =(0x74636166), ///< "fact"
  CKID_RIFF_LIST =(0x5453494c), ///< "LIST"
  CKID_RIFF_INFO =(0x4f464e49), ///< "INFO"
  CKID_RIFF_ISFT =(0x54465349), ///< "ISFT"
  CKID_RIFF_ICOP =(0x504F4349), ///< "ICOP"
  CKID_RIFF_IPRD =(0x44525049), ///< "IPRD"
  CKID_RIFF_ICRD =(0x44524349), ///< "ICRD"
  CKID_RIFF_IENG =(0x474e4549), ///< "IENG"
  CKID_RIFF_ICMT =(0x544D4349), ///< "ICMT"
  CKID_RIFF_IGNR =(0x524E4749), ///< "IGNR"
  CKID_RIFF_IKEY =(0x59454B49), ///< "IKEY"
  CKID_RIFF_IMED =(0x44454D49), ///< "IMED"
  CKID_RIFF_INAM =(0x4D414E49), ///< "INAM"
  CKID_RIFF_ISRC =(0x43525349), ///< "ISRC"
  CKID_RIFF_ITCH =(0x48435449), ///< "ITCH"
  CKID_RIFF_ISBJ =(0x4A425349), ///< "ISBJ"
  CKID_RIFF_ISRF =(0x46525349), ///< "ISRF"
  CKID_RIFF_DISP =(0x50534944)  ///< "DISP"
};

#define CHANNELS (1) // FOR ADPCMBLOCKHEADER


//============================================================================
//   OTHER CHUNKS
//   ------------
//   there is a "PAD " and a "JUNK" chunk both contain data to be thrown away
//   Only reason for 2 instead of 1 is because there are two rival companies.
//============================================================================

struct SampleWav::Impl::RiffHeader
{
  char      RIFF[4]; ///< magic number "RIFF"
  u_int32_t Size;    ///< Length of RIFF chunk.
};

//============================================================================
/// "fmt "
//============================================================================
struct SampleWav::Impl::FmtChunk
{
  u_int16_t FormatTag;      ///< format category
  u_int16_t nChannels;      ///< number of channels
  u_int32_t SamplesPerSec;  ///< Sampling rate
  u_int32_t AvgBytesPerSec; ///< for buffer estimation
  u_int16_t BlockAlign;     ///< data block size
  u_int16_t BitsPerSample;
  u_int16_t ExtraSize;      ///< size of extra data
};


//============================================================================
/// "cue "
/// before this there is a 32bit count of the number of cuepoints
//============================================================================
struct SampleWav::Impl::CuePoints
{
  int32_t Name;
  int32_t Position;
  char    Chunk[4];       ///< "slnt" or "data"
  int32_t ChunkStart;
  int32_t BlockStart;
  int32_t SampleOffset;
};

//============================================================================
/// before this there is a 32 bit count of the number of play lists
//============================================================================
struct SampleWav::Impl::PlayList
{
  int32_t Name;
  int32_t Length;
  int32_t Loops;
} PlayList;

//============================================================================
// MS ADPCM STRUCTS
//============================================================================
struct SampleWav::Impl::ADPCMCOEFSET
{
  int16_t iCoef1; // fixed point 8.8 values
  int16_t iCoef2;
};

struct SampleWav::Impl::ADPCMBLOCKHEADER
{
  u_int8_t  bPredictor[CHANNELS];  // nChannels is 1 0r 2
  u_int16_t iDelta[CHANNELS];
  u_int16_t iSamp1[CHANNELS];
  u_int16_t iSamp2[CHANNELS];
};

//============================================================================
// IMA ADPCM STRUCTS AND TABLES
//============================================================================
#pragma pack(1)

struct SampleWav::Impl::DVI_ADPCMBLOCKHEADER
{
  int16_t  Samp0;
  u_int8_t StepTableIndex;
  u_int8_t Reserved;
};

#pragma pack()

SampleWav::Impl::Impl(SampleWav& sample)
  : m_sample(sample),
    m_fact_file_size(0),
    m_comment(0),
    m_copyright(0),
    m_creation_date(0),
    m_engineer(0),
    m_genre(0),
    m_keywords(0),
    m_medium(0),
    m_product(0),
    m_subject(0),
    m_software(0),
    m_source(0),
    m_src_reference(0),
    m_technician(0),
    m_coefset(0)
{

}

SampleWav::Impl::~Impl()
{

}

//============================================================================

//============================================================================
bool SampleWav::Impl::GetRiffWaveHeader(FILE* fp, RiffHeader& riff)
{
  if (fread(&riff, sizeof(RiffHeader), 1, fp) != 1)
    {
      fclose(fp);
      return false;
    }
  if (strncmp((char *)riff.RIFF, "RIFF", 4))
    {
      fclose(fp);
      return false;
    }
  riff.Size += ftell(fp);

  u_int32_t chunkid; // four letter magic
  if (fread(&chunkid, 4, 1, fp) != 1)
    {
      fclose(fp);
      return false;
    }
  if (memcmp(&chunkid, "WAVE", 4))
    {
      fclose(fp);
      return false;
    }
  return true;
}

//============================================================================

//============================================================================
bool SampleWav::Impl::ReadFmtChunk(FILE*        fp,
                                   FmtChunk&    fmt,
                                   u_int32_t    chunksize)
{
  if (fread(&fmt, 16, 1, fp) != 1) // dont read in the extra size
    {
      fclose(fp);
      //"read error 2";
      return false;
    }

  m_sample.SetFrequency(fmt.SamplesPerSec);


  ppDebug1 ("Format ID     0x%x", fmt.FormatTag);
  ppDebug1 ("Channels      0x%x", fmt.nChannels);
  ppDebug1 ("SamplesPerSec %d",   fmt.SamplesPerSec);
  ppDebug1 ("BlockAlign    0x%x", fmt.BlockAlign);
  ppDebug1 ("BitsPerSample %d",   fmt.BitsPerSample);

  if (chunksize > 16) // the fmt chunk has extra data
    {
      if (fread(&(fmt.ExtraSize), sizeof(u_int16_t), 1, fp) != 1)
        {
          if (ferror(fp))
            {
              fclose(fp);
              //m_error_message = "read error 4";
              return false;
            }
        }

      ppDebug1 ("  Extra size: %d", fmt.ExtraSize);

      if ((unsigned)fmt.ExtraSize + 18 != chunksize)
        {
          // corruption?
          ppWarning ("Fmt Chunk has incorrect extra size");
          return false;
        }

      if (fread(&fmt+18, fmt.ExtraSize, 1, fp) != 1)
        {
          if (ferror(fp))
            {
              fclose(fp);
              //m_error_message = "read error 4";
              return false;
            }
        }



      u_int16_t     SamplesPerBlock;

      // process format specific extra fmt data

      if (fmt.FormatTag == WAVE_FORMAT_ADPCM)
        {
          // retrieve Samples Per Block
          SamplesPerBlock = (u_int16_t)(*(u_int16_t*)(&fmt+18));

          /* adpcmcoefset points to NumCoef ADPCMCOEFSET's */
          u_int16_t     NumCoef = (u_int16_t)(*(u_int16_t*)(&fmt+18+2));


          // allocate memory for coefset
          m_coefset = new ADPCMCOEFSET [NumCoef];
          if (!m_coefset)
            {
              fclose(fp);
              //m_error_message = "malloc error";
              return false;
            }
          memcpy(m_coefset, &fmt+18+4, NumCoef*sizeof(ADPCMCOEFSET));
        }
      else if (fmt.FormatTag == WAVE_FORMAT_DVI_ADPCM)
        {
          // retrieve Samples Per Block
          SamplesPerBlock = (u_int16_t)(*(u_int16_t*)(&fmt+18));
        }
    }
  else
    {
      fmt.ExtraSize = 0;
    }
  return true;
}

bool SampleWav::Impl::ReadDataChunk(FILE*           fp,
                                    const FmtChunk& fmt,
                                    u_int32_t       chunksize)
{
  m_sample.SetLength(chunksize);

  switch (fmt.FormatTag)
    { // Format switch
    case WAVE_FORMAT_PCM:
      {
        /* READ IN SAMPLE DATA */
        // should be 8 or 16 bit
        if (fmt.BitsPerSample == 16)
          m_sample.Set16Bit(); /* set 16 bit flag */
        else if (fmt.BitsPerSample == 8)
          m_sample.Set8Bit();


        u_int8_t* temp =  new u_int8_t [chunksize];
        if (!temp)
          {
            fclose(fp);
            //m_error_message = "Memory Allocation Error";
            return false;
          }
        if (fread(temp, chunksize, 1, fp)!=1)
          {
            if (ferror(fp))
              {
                fclose(fp);
                //m_error_message = "read error data 1";
                return false;
              }
          }

        if (fmt.nChannels == 1)
          {
            m_sample.m_data0 = temp;
          }
        else if (fmt.nChannels == 2)
          {
            // need to worry about sample size
            m_sample.SetLength(chunksize/2);
            m_sample.SetStereo();
            m_sample.m_data0 = new u_int8_t [m_sample.GetLength()];
            m_sample.m_data1 = new u_int8_t [m_sample.GetLength()];
            for (Sample::position_t i = 0;
                 i < m_sample.GetLength();++i)
              {
                if (fmt.BitsPerSample == 16)
                  {
                    m_sample.m_data0[i] = temp[i*2];
                    m_sample.m_data0[i] = temp[i*2+1];
                    ++i;
                    m_sample.m_data1[i] = temp[i*2];
                    m_sample.m_data1[i] = temp[i*2+1];
                  }
                else
                  {
                    m_sample.m_data0[i] = temp[i*2];
                    m_sample.m_data1[i] = temp[i*2+1];
                  }
              }
            delete [] temp;
          }
      }
      break;
    case WAVE_FORMAT_ALAW:
      {
        u_int8_t *temp_data = new u_int8_t [m_sample.GetLength()];
        if (!temp_data)
          {
            //m_error_message = "malloc error";
            return false;
          }
        if (fread(temp_data, m_sample.GetLength(), 1, fp)!=1)
          {
            if (ferror(fp))
              {
                //m_error_message = "read error temp_data";
                return false;
              }
          }

        // multiply length by two (as samples are 16 bit)
        m_sample.SetLength(m_sample.GetLength()*2);
        m_sample.Set16Bit();
        m_sample.m_data0 = new u_int8_t [m_sample.GetLength()];
        if (!m_sample.m_data0)
          {
            //m_error_message = "malloc error";
            return false;
          }
        Codec::DecodeALaw(temp_data, (u_int16_t *&)m_sample.m_data0, m_sample.GetSamples());

        break;
      }
    case WAVE_FORMAT_MULAW:
      {
        u_int8_t* temp_data = new u_int8_t [m_sample.GetLength()];
        if (!temp_data)
          {
            //m_error_message = "malloc error";
            return false;
          }
        if (fread(temp_data, m_sample.GetLength(), 1, fp)!=1)
          {
            if (ferror(fp))
              {
                //m_error_message = "read error temp_data";
                return false;
              }
          }

        // multiply length by two (as samples are 16 bit)
        m_sample.SetLength(m_sample.GetLength()*2);
        m_sample.Set16Bit();
        m_sample.m_data0 = new u_int8_t [m_sample.GetLength()];
        if (!m_sample.m_data0)
          {
            //m_error_message = "malloc error";
            return false;
          }

        Codec::DecodeMuLaw(temp_data, (u_int16_t *&)m_sample.m_data0, m_sample.GetSamples());

        break;
      }
    case WAVE_FORMAT_ADPCM:
      {
        // I haven't been able to find the tables microsoft uses for
        // their adpcm format so this is still unfinished
        /* VARIABLES USED FOR MS ADPCM */
        ADPCMBLOCKHEADER blockheader;
        u_int8_t*     adpcmdata = 0;

        /* read the ADPCM BLOCK HEADER */
        if (fread(&blockheader, sizeof(ADPCMBLOCKHEADER), 1, fp)!=1)
          {
            if (ferror(fp))
              {
                perror("fread");
                fclose(fp);
                //m_error_message = "read error blockheader";
                return false;
              }
          }
        adpcmdata = new u_int8_t [chunksize];
        if (!adpcmdata)
          {
            //m_error_message = "malloc error";
            return false;
          }
        /* READ IN ENCODED DATA */
        if (fread(adpcmdata, chunksize, 1, fp) != 1)
          {
            if (ferror(fp))
              {
                perror("fread");
                //m_error_message = "read error adpcmdata";
                return false;
              }
          }
        m_sample.m_data0 = new u_int8_t [m_fact_file_size];
        if (!m_sample.m_data0)
          {
            //m_error_message = "malloc error";
            return false;
          }

        //  lPredSamp = ((iSamp1 * iCoef1) +
        //  (iSamp2 *iCoef2)) / FIXED_POINT_COEF_BASE

        // DECODE IT! real sample length is given in a Fact chunk
        delete [] adpcmdata;
        adpcmdata = 0;
        //m_error_message = "NOT IMPLEMENTED YET!!";
        return false;
      }

    case WAVE_FORMAT_DVI_ADPCM:
      {
        // VARIABLES USED FOR DVI_ADPCM
        DVI_ADPCMBLOCKHEADER dviblockheader;
        u_int8_t        Delta;       // only use 4bits
        long        valprev;
        long        valpred;
        int         StepTableIndex; // Step Table Index
        long        blocklength;
        u_int8_t*         BlockData = 0;
        u_int8_t*         DataPointer = 0;
        unsigned int      DataRead;


        // Read in each block and decode it until all
        // blocks are read
        // allocate memory for Sample Data

        m_sample.SetLength(m_fact_file_size*2); // Taken from fact chunk
        m_sample.m_data0 = new u_int8_t [m_sample.GetLength()];
        DataPointer = m_sample.m_data0;
        // allocate memory for BlockData
        // subtract 4 from length to get the length of the data
        blocklength = fmt.BlockAlign - sizeof(DVI_ADPCMBLOCKHEADER);
        BlockData = new u_int8_t [blocklength];
        if (!BlockData)
          {
            //m_error_message = "malloc error";
            return false;
          }
        DataRead = 0;
        while (DataRead < chunksize)
          {
            if (fread(&dviblockheader,
                      sizeof(DVI_ADPCMBLOCKHEADER),
                      1,
                      fp)!=1)
              {
                if (ferror(fp))
                  {
                    perror("fread");
                    //m_error_message = "read error dviblockheader";
                    return false;
                  }
              }
            /* Output Samp0 */
            *DataPointer++ = dviblockheader.Samp0 & 0xff;
            *DataPointer++ = ((dviblockheader.Samp0 >> 8) & 0xff);

            valprev = dviblockheader.Samp0;
            StepTableIndex = dviblockheader.StepTableIndex;

            DataRead += fmt.BlockAlign;
            /* Read in Block Data */
            if (fread(BlockData, blocklength, 1, fp) != 1)
              {
                if (ferror(fp))
                  {
                    perror("fread");
                    //m_error_message = "read error blockdata";
                    return false;
                  }
              }
            for (int i = 0; i < blocklength<<1; ++i)
              {
                /* while still Samples to decode */
                /* Get Next Sample Code SampXCode is only 4 bits !*/
                if (i % 2)
                  Delta = (BlockData[i >> 1] >> 4) & 0xf;
                else
                  Delta =  BlockData[i >> 1] & 0xf;

                /* Calculate the new sample */
                valpred = Codec::DecodeAdpcmSample(Delta, valprev, StepTableIndex);
                /* Output new Sample - SampX */
                *DataPointer++ = valpred & 0xff;
                *DataPointer++ = ((valpred >> 8) & 0xff);
                /* Save the previous sample */
                valprev = valpred;
              }
          }
        // sample data has been converted to 16 bit signed
        m_sample.Set16Bit(); // 16 bit
        break;
      }
    case WAVE_FORMAT_UNKNOWN:
    default:
      fclose(fp);
      //m_error_message = "Unknown Format";
      return false;
    }
  return true;
}

bool SampleWav::Impl::ReadInfoChunk(FILE* fp, u_int32_t chunkid, u_int32_t chunksize)
{
#define CASE_INFO(ID, NAME, VARIABLE) \
 case ID:  \
     printf(NAME" : %s\n", xchar); \
     VARIABLE = xchar; \
     xchar = 0; \
     break;

  // read in 0-terminated string and pad
  char* xchar = new char [chunksize + (chunksize % 2)];
  if (!xchar)
    {
      fclose(fp);
      //m_error_message = "malloc error";
      return false;
    }
  fread(xchar, 1, chunksize + (chunksize % 2), fp);
  switch(chunkid)
    {
    case CKID_RIFF_INAM:
      m_sample.SetName(xchar);
      ppDebug1 ("  Name       : %s\n", xchar);
      break;

      CASE_INFO(CKID_RIFF_ICMT, "Comment      ", m_comment)
        CASE_INFO(CKID_RIFF_ICOP, "Copyright    ", m_copyright)
        CASE_INFO(CKID_RIFF_ICRD, "Creation Date", m_creation_date)
        CASE_INFO(CKID_RIFF_IENG, "Engineer     ", m_engineer)
        CASE_INFO(CKID_RIFF_IGNR, "Genre        ", m_genre)
        CASE_INFO(CKID_RIFF_IKEY, "Keywords     ", m_keywords)
        CASE_INFO(CKID_RIFF_IMED, "Medium       ", m_medium)
        CASE_INFO(CKID_RIFF_IPRD, "Product      ", m_product)
        CASE_INFO(CKID_RIFF_ISBJ, "Subject      ", m_subject)
        CASE_INFO(CKID_RIFF_ISFT, "Software     ", m_software)
        CASE_INFO(CKID_RIFF_ISRC, "Source       ", m_source)
        CASE_INFO(CKID_RIFF_ISRF, "Src Reference", m_src_reference)
        CASE_INFO(CKID_RIFF_ITCH, "Technician   ", m_technician)

        default:
          ppWarning ("  Unknown INFO chunk 0x%X", chunkid);
    }
  if (xchar)
    {
      delete [] xchar;
      xchar = 0;
    }
  return true;
}

bool SampleWav::Impl::ReadListChunk(FILE*     fp,
                                    u_int32_t chunksize)
{
  u_int32_t chunkid;
  if (fread(&chunkid, 1, 4, fp) != 4)
    {
      fclose(fp);
      //  "read error LIST 1";
      return false;
    }

  u_int32_t listsize; // size of INFO list
  long      listseek;
  // LISTs have an list type after their size
  if (chunkid != CKID_RIFF_INFO)
    {
      ppWarning ("  Unknown LIST type");
      return false;
    }
  // NO ChunkSize after "INFO"
  listsize = chunksize  - 4 + ftell(fp);
  listseek = ftell(fp);
  while (listseek < (long)listsize)
    {
      fseek(fp, listseek, SEEK_SET);
      if (fread(&chunkid, sizeof(char), 4,fp) != 4)
        {
          fclose(fp);
          //m_error_message = "read error INFO chunkid";
          return false;
        }
      fread(&chunksize, 4, 1, fp);
      listseek = chunksize + ftell(fp) + chunksize % 2;
      ReadInfoChunk(fp, chunkid, chunksize);
    }
  return true;
}

//============================================================================
//
//============================================================================
bool SampleWav::Impl::ReadChunk(FILE*      fp,
                                u_int32_t  chunkid,
                                u_int32_t& chunksize,
                                FmtChunk*& fmt)
{
  switch (chunkid)
    {
    case CKID_RIFF_FMT: /* "fmt "*/
      // allocate space for the fmt chunk
      // different formats have different size fmt chunks
      if (fmt != 0)
        {
          // more than one fmt chunk?
        }
      fmt = (FmtChunk*)new char [chunksize];
      memset(fmt, 0, chunksize);
      ReadFmtChunk(fp, *fmt, chunksize);
      break;

    case CKID_RIFF_DATA: /* "data" */
      if (chunksize & 1)
        ++chunksize; // must be an even number
      if (!fmt)
        {
          // We need to have read the fmt chunk before
          // we read the data chunk
          return false;
        }
      ReadDataChunk(fp, *fmt, chunksize);
      break;

    case CKID_RIFF_DISP:
      {
        // read in 0-terminated string and pad
        char* xchar = new char [chunksize + (chunksize % 2)];
        if (!xchar)
          {
            fclose(fp);
            //m_error_message = "malloc error";
            return false;
          }
        if (fread(xchar, chunksize + (chunksize % 2),1, fp) != 1)
          {
            perror("fread");
            //m_error_message = "read error xchar";
            return false;
          }
      }
      break;

    case CKID_RIFF_FACT: // "fact"
      if (fread(&m_fact_file_size, sizeof(u_int32_t), 1, fp) != 1)
        {
          fclose(fp);
          //m_error_message = "read error FACT";
          return false;
        }
      ppDebug1 ("FACT FileSize %d", m_fact_file_size);
      break;

    case CKID_RIFF_LIST: // "LIST"
      ReadListChunk(fp, chunksize);
      break;

    default:
      ppWarning ("  Unknown Chunk : %4s  0x%X", (char*)&chunkid, chunkid);

      return false;

    } /*END SWITCH*/
  return true;
}

//============================================================================
//
//============================================================================
bool SampleWav::Impl::ReadInChunks(FILE* fp, RiffHeader& riff)
{
  long nextseek = ftell(fp);

  FmtChunk* fmt = 0;

  while (nextseek < (long)riff.Size)
    {

      if (fseek(fp, nextseek, SEEK_SET))
        {
          // FIX THIS UP!
          ppInternalError ("seek error");
        }

      u_int32_t chunkid;

      if (fread(&chunkid, sizeof(char), 4, fp) != 4)
        {
          fclose(fp);
          // "read error chunkid";
          return false;
        }

      u_int32_t chunksize;

      if (fread(&chunksize, 4, 1, fp) != 1)
        {
          fclose(fp);
          // "read error chunksize";
          return false;
        }

      ppDebug1 ("CHUNK %.4s length %d", (char*)&chunkid, chunksize);

      nextseek = chunksize + ftell(fp); // get position of next chunk

      // read the chunk and process it -- ignore chunks we don't know
      ReadChunk(fp, chunkid, chunksize, fmt);
    }
  delete fmt;
  return true;
}
//============================================================================
// constructors and destructors
//============================================================================
SampleWav::SampleWav()
{

}


SampleWav::~SampleWav()
{

}

//============================================================================
/**

 */
//============================================================================
bool SampleWav::Load(const char *filename)
{
  Impl Impl(*this);

  Impl::RiffHeader Riff;

  FILE* fp = fopen(filename, "rb");

  if (!fp)
    return false;

  SetName(filename); // let the default name be the filename

  if (!Impl.GetRiffWaveHeader(fp, Riff))
    return false;

  if (!Impl.ReadInChunks(fp, Riff))
    return false;

  fclose(fp);


  return true;
}


bool SampleWav::Save(const char* filename, const Sample& sample) const
{
  return false;
}

PP_I_NAMESPACE_END
PP_NAMESPACE_END
