/*
 * C API of the PenguinPlay File Access system
 * Part of the PenguinPlay File handling system
 *
 * Author: Christian Reiniger <warewolf@mayn.de>
 *
 * License: See the accompanying file LICENSE
 *
 * Last change by $Author: creinig $
 * on             $Date: 1999/08/29 19:58:59 $
 * to             $Revision: 1.4 $
 *
 */


#ifndef _PENGUINFILE_H
#define _PENGUINFILE_H 1

#include <PenguinPlay/PenguinPlay.h>

#ifdef PP_CPLUSPLUS
#  include <cstdio>
#  include <cerrno>
#else
#  include <stdio.h>
#  include <errno.h>
#endif


#if HAVE_UNISTD_H
#  include <dirent.h>
#  include <sys/types.h>
#else
#  error "Sorry - POSIX functionality is needed"
#endif



/*
 * Some Maximum values -------------------------------------------------------
 */


#define PPF_MAX_FILENAME_LENGTH  255  /* max. length of filenames (w/o '\0') */

#define PPF_MAX_URL_FS_LENGTH     32  /* max. length of fstype specifiers
                                         (w/o '\0') */

#define PPF_MAX_MOUNT_COUNT       64  /* max number of file systems that can
                                         be mounted at the same time */

#define PPF_MAX_PATH_DEPTH        32  /* max. number of elements in a path */




#ifdef PP_CPLUSPLUS
extern "C"
{
#endif



typedef enum
{
	ppfFS_undefined = 0,
	ppfFS_plain,
	ppfFS_pak,
	ppfFS_file,
	ppfFS_virt,
	ppfFS_ftp,      /* not supported yet */
	ppfFS_http,     /* not supported yet */
	ppfFS_numvals   /* how many FSTypes do we know? */
} ppfFSType;



typedef enum
{
	ppfFT_undefined = 0,
	ppfFT_file,
	ppfFT_directory,
	ppfFT_symlink        /* not supported yet */
} ppfFileType;



/* To change without notice. Do *NOT* use numeric values directly. */
typedef enum
{
	ppfFA_none         =  0,
	ppfFA_compressed   =  1,
	ppfFA_encrypted    =  2,
	ppfFA_readable     =  4,
	ppfFA_writeable    =  8,
	ppfFA_executable   = 16, /* "searchable" for directories */
	ppfFA_construction = 32  /* for PakFiles being constructed. Means
	                          * that only files/dirs of matching type
                                  * may be added, and only via ppfCopy () */
} ppfFileAttrib;




typedef enum
{
	ppfMF_none             =  0,
	ppfMF_write            =  1,
	ppfMF_override_files   =  2,
	ppfMF_override_dirs    =  4,
	ppfMF_override_radical =  8,
	ppfMF_force            = 16
} ppfMountFlags;


/* Collection of flags to be used for a great range of functions
 * (right now only ppfCopy () uses them) */
typedef enum
{
	ppfGF_none              = 0,
	ppfGF_recursive         = 1,
	ppfGF_follow_links      = 2,
	ppfGF_abort_on_problems = 4
} ppfGenericFlags;




/* Our custom file handle structure. Do *not* access any internals of it, as
 * they will change for sure - without notice */
typedef struct
{
	ppfFSType  FS;      /* What kind of file is it? */
	FILE      *Handle;  /* stdio filehandle of the file itself
	                       or its parent PakFile */
	ppSizeT    DSize;   /* its data size in bytes */
	ppSizeT    FSize;   /* its real size in bytes */
	ppOffsetT  SiP;     /* its starting position in PakFile */
	ppOffsetT  RPos;    /* reading pos (currently RPos == WPos) */
	ppOffsetT  WPos;    /* writing pos (currently RPos == WPos) */
	pp32       Attribs; /* access flags, subset of ppfFileAttrib */
	int        LastErr; /* Code of last error (-> errno) */
	void      *File;    /* Pointer to the respective ppf_File
	                       class. Has to be void because we cannot
                               use pointers to classes in a C header */
} ppFILE;




/* All fields in this struct can be used directly (in fact, you *should* do
 * exactly that ;) */
typedef struct
{
	const char  *Name;    /* length <= PP_MAX_FILENAME_LENGTH */
	ppTimeT      CTime;   /* Creation Time */
	ppTimeT      MTime;   /* Modification Time */
	pp32         Attribs; /* Attributes. Use ppfFileAttrib to evaluate */
	ppSizeT      Size;    /* File size */
	ppfFileType  Type;    /* what kind of file is it? */
	ppfFSType    FS;      /* in what kind of file system is it? */
} ppfDirEntry;




/* for internal use only */
typedef struct
{
	/* Where is the dir? */
        ppfFSType    FS;

        /* Number of current Entry (for ppfSeekDir etc) */
	ppOffsetT    EntryNo;

	/* Dirname & Path (for stat() calls in ppf_ReadDirPlain ()) */
	const char  *BasePath;

	/* The struct returned by ppf_ReadDir() */
        ppfDirEntry *DirEntry;

	/* Pointers to the respective ppf_DirEntry objects. They have to be
	 * (void *) because we cannot use pointers to C++ objects in a
	 * C header */
	const void  *First;
	const void  *Current;

	/* POSIX DIR structure for "normal" dirs */
	DIR         *PosixDir;
} ppfDIR;











/*
 * The Procedural API. This part is a 95% stdio clone, so everyone should be
 * familiar with it.
 */


ppFILE   *ppfOpen     (const char *URL, const char *mode);
ppFILE   *ppfReopen   (const char *URL, const char *mode, ppFILE *stream);
int       ppfClose    (ppFILE *stream);


int       ppfFlush    (ppFILE *stream);
void      ppfSetBuf   (ppFILE *stream, char *buf);
int       ppfSetVBuf  (ppFILE *stream, char *buf, int mode, ppSizeT size);

/* ppfPrintF   */
/* ppfVFPrintF */
/* ppfScanF    */

int       ppfGetC     (       ppFILE *stream);
int       ppfPutC     (int c, ppFILE *stream);
int       ppfUnGetC   (int c, ppFILE *stream);
char     *ppfGetS     (      char *s, int n, ppFILE *stream);
int       ppfPutS     (const char *s,        ppFILE *stream);

ppSizeT   ppfRead     (      void *ptr, ppSizeT size, ppSizeT nobj, ppFILE *stream);
ppSizeT   ppfWrite    (const void *ptr, ppSizeT size, ppSizeT nobj, ppFILE *stream);

int       ppfSeek     (ppFILE *stream, ppOffsetT offset, int origin);
ppOffsetT ppfTell     (ppFILE *stream);
void      ppfRewind   (ppFILE *stream);
int       ppfGetPos   (ppFILE *stream,       ppOffsetT *ptr);
int       ppfSetPos   (ppFILE *stream, const ppOffsetT *ptr);


void      ppfClearErr (ppFILE *stream);
int       ppfEof      (ppFILE *stream);
int       ppfError    (ppFILE *stream);
void      ppfPError   (const char *s);



/* Directory reading. An almost complete clone of the POSIX directory API */

ppfDIR      *ppfOpenDir   (const char *URL);
int          ppfCloseDir  (ppfDIR *Dir);
ppfDirEntry *ppfReadDir   (ppfDIR *Dir);
void         ppfSeekDir   (ppfDIR *Dir, ppOffsetT offset);
ppOffsetT    ppfTellDir   (ppfDIR *Dir);
void         ppfRewindDir (ppfDIR *Dir);

/* These use the normal ppfDirEntry structure as "stat" replacement
 * (it's completely sufficient for this task) */
int          ppfStat      (const char *URL, ppfDirEntry *dire);
int          ppfLStat     (const char *URL, ppfDirEntry *dire);


/* Misc. other POSIX clone functions */

int          ppfChdir     (const char *Path);
int          ppfMkdir     (const char *URL, int Openmode);
int          ppfChmod     (const char *URL, int Mode);



/* PenguinFile specific stuff */

/* return 0 on success and != 0 on failure */
int          ppfMount     (const char *What, const char *Where, int Flags);
int          ppfUmount    (const char *URL, int Flags);
int          ppfCreatePak (const char *URL, int Format,
			   const char *GameID, const char *VendorID,
			   pp32 GameVerMajor, pp32 GameVerMinor);

/* Right now this is only a very primitive prototype.
 * Only one file can be copied. The "Flags" parameter is completely ignored.
 * "Src" has to be the URL of the source file.
 * "Dest" has to be the *qualified* (i.e. with fstype specifier) URL of the
 * target *file* (i.e. *not* the destination directory).
 * Only "plain" and "pak" are supported as destination filesystems.
 */
int          ppfCopy      (const char *Src, const char *Dest, int Flags);

#ifdef PP_CPLUSPLUS
}
#endif

#endif
