/*
 * Debugging macros & functions
 * #included by PenguinPlay.h - please do not include directly
 *
 * Project : PenguinPlay
 * Part    : Generic
 * Authors : Christian Reiniger <warewolf@mayn.de>
 *           Adrian Ratnapala <raka@mailhost.bit.net.au>
 * License : See the accompanying file LICENSE
 *
 * Last change by $Author$
 * on             $Date$
 * to             $Revision$
 *
 */


#ifndef _PP_DEBUG_H
#define _PP_DEBUG_H


/*
 * This is used in the debugging versions of the messaging
 * functions.
 */
#ifdef __GNUC__
  #define PP_FUNCTION_NAME __PRETTY_FUNCTION__
#else
  extern const char* _ppNoFuncNameMsg;
  #define PP_FUNCTION_NAME _ppNoFuncNameMsg
#endif





/*****************************************************************************
 *
 * Definition of the Debuglevels:
 *
 * PP_DEBUGLEVEL == 0: No debugging info, ppFatalError () macros *are*
 *                     evaluated (i.e. they should be only used for *really*
 *                     critical stuff). Exits the program.
 *
 * PP_DEBUGLEVEL >= 1: Evaluation of ppInternalError () (new macro, used for
 *                     internal return value checking etc, i.e. stuff that can
 *                     be ommitted in the final, extensively debugged version)
 *                     Exits the program.
 *
 * PP_DEBUGLEVEL >= 2: Evaluation of ppWarning (). Debugging info warning the
 *                     debugger about oddities that shouldn't be (but are not
 *                     critical)
 *                     Evaluation of ppError (). Debugging info informing the
 *                     debugger about a serious error that is most likely
 *                     not caused by PFile itself. Exits the program.
 *
 * PP_DEBUGLEVEL >= 3: Evaluation of ppAssert () and ppThrow ()
 *
 * PP_DEBUGLEVEL >= 4: Evaluation of ppDebug1 (). Info about entering/leaving
 *                     major code sections (e.g. "configfile parsing started"
 *                     / "configfile parsing finished successfully")
 *
 * PP_DEBUGLEVEL >= 5: Evaluation of ppDebug2 (). More detailed progress info
 *                     (e.g. "Processing configfile line x")
 *
 * PP_DEBUGLEVEL >= 6: Evaluation of ppDebug3 (). medium detail info about
 *                     the processed data (e.g. "Line contains token
 *                     'DefaultResolution' with value '800x600'")
 *
 * PP_DEBUGLEVEL >= 7: Evaluation of ppDebug4 (). Info about variable values
 *                     at important points (e.g.  "(int) DefaultResX = 800  ;
 *                     (int) DefaultResY = 600")
 *
 * PP_DEBUGLEVEL >= 8: Evaluation of ppDebug5 (). Really excessively detailed
 *                     information (e.g. "LoopVariable = 1376  ;
 *                     NoOfCopiedBytes = 327860 ; LastByteVal = 0x3F")
 *
 * Of course That doesn't mean that every piece of code has to give this much
 * debug info ;)  But if debug info is given it has to behave this way.
 *
 *
 *
 * Some Notes from Adrian about the code:
 *
 * ----------------------------
 *  #define ppDebug (\
 *      *_pp_db_lineno_loc()=__LINE__,\
 *      *_pp_db_file_loc()=__FILE__,\
 *      *_pp_db_func_loc()=_PP_FUNCTION_NAME,\
 *       _ppDbgDebug\
 *  )
 * --------------------
 *
 * This is an evil C trick to use  the comma operator.  It
 * sets a few "global variables" to appropriate values, and evaluates
 * to a pointer _ppDbgDebug.
 *
 * Thus :
 *   ppDebug(SOME ARGS)
 *
 * becomes equivilent to:
 *   Set some global variables to display the source location;
 *   _ppDbgDebug(SOME ARGS);
 *
 * On the other hand, when debugging is off, we get
 *   (void)(SOME ARGS);
 * This causes, the args to be evaluated if they have side effects,
 * but does not print anything out, since they are just ignored by
 * the cast to void.  (Again, the comma operator comes into play).
 *
 * The _pp_db_xxxx__loc() functions are used instead of straight globals
 * so we can fix them to be thread safe (this works like errno in libc).
 *
 *
 */



/* For all debugging macros */
PP_EXTC int* _pp_db_lineno_loc(void);
PP_EXTC const char** _pp_db_file_loc(void);
PP_EXTC const char** _pp_db_func_loc(void);


/* Always evaluated */
PP_EXTC void _ppFatalError(const char* msg, ...) PP_PRINTF_STYLE(1,2);

#define ppFatalError (\
     *_pp_db_lineno_loc()=__LINE__,\
     *_pp_db_file_loc()=__FILE__,\
     *_pp_db_func_loc()=PP_FUNCTION_NAME,\
      _ppFatalError\
)



#if PP_DEBUGLEVEL >= 1
  PP_EXTC void _ppInternalError(const char* msg, ...) PP_PRINTF_STYLE(1,2);

  #define ppInternalError (\
       *_pp_db_lineno_loc()=__LINE__,\
       *_pp_db_file_loc()=__FILE__,\
       *_pp_db_func_loc()=PP_FUNCTION_NAME,\
        _ppInternalError\
  )
#else
  #define ppInternalError (void)
#endif


#if PP_DEBUGLEVEL >= 2
  PP_EXTC void _ppWarning (const char *msg, ...) PP_PRINTF_STYLE(1,2);
  PP_EXTC void _ppError   (const char *msg, ...) PP_PRINTF_STYLE(1, 2);

  #define ppWarning (\
      *_pp_db_lineno_loc()=__LINE__,\
      *_pp_db_file_loc()=__FILE__,\
      *_pp_db_func_loc()=PP_FUNCTION_NAME,\
       _ppWarning\
  )

  #define ppError (\
      *_pp_db_lineno_loc()=__LINE__,\
      *_pp_db_file_loc()=__FILE__,\
      *_pp_db_func_loc()=PP_FUNCTION_NAME,\
       _ppError\
  )
#else
#  define ppWarning (void)
#  define ppError   (void)
#endif


#if PP_DEBUGLEVEL >= 3
  #define ppAssert(expr)  (\
      *_pp_db_lineno_loc()=__LINE__,\
      *_pp_db_file_loc()=__FILE__,\
      *_pp_db_func_loc()=PP_FUNCTION_NAME,\
	((expr) ? (void) (0) : _ppInternalError ("Assertion " #expr " failed"))\
  )

  #define ppThrow(XCept, details) (\
    throw (XCept (PP_FUNCTION_NAME, details, __STRING(__LINE__), __FILE__)) \
  )
#else
  #define ppAssert(expr) (void) ()
  #define ppThrow(XCept, details) (void) ()
#endif


#if PP_DEBUGLEVEL >= 4
  PP_EXTC void _ppDebug(const char* msg, ...) PP_PRINTF_STYLE(1,2);

  #define ppDebug1 (\
      *_pp_db_lineno_loc()=__LINE__,\
      *_pp_db_file_loc()=__FILE__,\
      *_pp_db_func_loc()=PP_FUNCTION_NAME,\
       _ppDebug\
  )

  #define ppDebug ppDebug1 // DEPRECATED ! Use ppDebugx () instead

#else
  #define ppDebug1 (void)
  #define ppDebug  (void)  // DEPRECATED ! Use ppDebugx () instead
#endif


#if PP_DEBUGLEVEL >= 5
  #define ppDebug2 ppDebug
#else
  #define ppDebug2 (void)
#endif


#if PP_DEBUGLEVEL >= 6
  #define ppDebug3 ppDebug
#else
  #define ppDebug3 (void)
#endif


#if PP_DEBUGLEVEL >= 7
  #define ppDebug4 ppDebug
#else
  #define ppDebug4 (void)
#endif


#if PP_DEBUGLEVEL >= 8
  #define ppDebug5 ppDebug
#else
  #define ppDebug5 (void)
#endif




/*
 * ------------------------------------------------------------ Debug Messages
 *
 * FIX: Do not use these yet - They will most likely be completely rewritten
 */



#define PP_ERROR_INTERNAL         ppInternalEror ("Unknown internal PenguinPlay error")

#define PP_WARN_NEW_FAIL          ppWarning ("Out of memory")
#define PP_WARN_INVALID_ARG       ppWarning ("Invalid argument")
#define PP_WARN_INVALID_PATH      ppWarning ("Invalid path")
#define PP_WARN_INVALID_FILE      ppWarning ("Non-existant file specified in path")
#define PP_WARN_INVALID_DIR       ppWarning ("Non-existant directory specified in path")
#define PP_WARN_INVALID_FS        ppWarning ("Invalid file-system specifier")
#define PP_WARN_NO_CONSTRUCT      ppWarning ("Object could not be constructed")
#define PP_WARN_NO_COMPONENT_INIT ppWarning ("Could not initialise component")




#endif