/*
    pygame - Python Game Library
    Copyright (C) 2000-2001  Pete Shinners

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Pete Shinners
    pete@shinners.org
*/

/*
 *  drawing module for pygame
 */
#include "pygame.h"
#include "draw.h"



    /*DOC*/ static char doc_line[] =
    /*DOC*/    "pygame.draw.line(Surface, color, startpos, endpos) -> Rect\n"
    /*DOC*/    "draw a line on a surface\n"
    /*DOC*/    "\n"
    /*DOC*/    "Draws a line on a surface. This will respect the clipping\n"
    /*DOC*/    "rectangle. A bounding box of the effected area is returned\n"
    /*DOC*/    "as a rectangle.\n"
    /*DOC*/    "\n"
    /*DOC*/    "The color argument can be either a RGB sequence or mapped color integer.\n"
    /*DOC*/    "\n"
    /*DOC*/    "This function will temporarily lock the surface.\n"
    /*DOC*/ ;

static PyObject* line(PyObject* self, PyObject* arg)
{
	PyObject *surfobj, *colorobj, *start, *end;
	SDL_Surface* surf;
	short startx, starty, endx, endy;
	int top, left, bottom, right;
	int pts[4];
	Uint8 rgba[4];
	Uint32 color;
	int anydraw;

	/*get all the arguments*/
	if(!PyArg_ParseTuple(arg, "O!OOO", &PySurface_Type, &surfobj, &colorobj, &start, &end))
		return NULL;
	surf = PySurface_AsSurface(surfobj);

	if(surf->format->BytesPerPixel <= 0 || surf->format->BytesPerPixel > 4)
		return RAISE(PyExc_ValueError, "unsupport bit depth for line draw");

	if(PyInt_Check(colorobj))
		color = (Uint32)PyInt_AsLong(colorobj);
	else if(RGBAFromObj(colorobj, rgba))
		color = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
	else
		return RAISE(PyExc_TypeError, "invalid color argument");

	if(!TwoShortsFromObj(start, &startx, &starty))
		return RAISE(PyExc_TypeError, "Invalid start position argument");
	if(!TwoShortsFromObj(end, &endx, &endy))
		return RAISE(PyExc_TypeError, "Invalid end position argument");
	

	if(!PySurface_Lock(surfobj)) return NULL;

	pts[0] = startx; pts[1] = starty;
	pts[2] = endx; pts[3] = endy;
	anydraw = clip_and_draw_line(surf, &surf->clip_rect, color, pts);

	if(!PySurface_Unlock(surfobj)) return NULL;


	/*compute return rect*/
	if(!anydraw)
		return PyRect_New4(startx, starty, 0, 0);
	if(pts[0] < pts[2])
	{
		left = pts[0];
		right = pts[2];
	}
	else
	{
		left = pts[2];
		right = pts[0];
	}
	if(pts[1] < pts[3])
	{
		top = pts[1];
		bottom = pts[3];
	}
	else
	{
		top = pts[3];
		bottom = pts[1];
	}
	return PyRect_New4((short)left, (short)top, (short)(right-left+1), (short)(bottom-top+1));
}


    /*DOC*/ static char doc_lines[] =
    /*DOC*/    "pygame.draw.lines(Surface, color, closed, point_array) -> Rect\n"
    /*DOC*/    "draw multiple connected lines on a surface\n"
    /*DOC*/    "\n"
    /*DOC*/    "Draws a sequence on a surface. You must pass at least two points\n"
    /*DOC*/    "in the sequence of points. The closed argument is a simple boolean\n"
    /*DOC*/    "and if true, a line will be draw between the first and last points.\n"
    /*DOC*/    "\n"
	/*DOC*/    "This will respect the clipping rectangle. A bounding box of the\n"
	/*DOC*/    "effected area is returned as a rectangle.\n"
    /*DOC*/    "\n"
    /*DOC*/    "The color argument can be either a RGB sequence or mapped color integer.\n"
    /*DOC*/    "\n"
    /*DOC*/    "This function will temporarily lock the surface.\n"
    /*DOC*/ ;

static PyObject* lines(PyObject* self, PyObject* arg)
{
	PyObject *surfobj, *colorobj, *closedobj, *points, *item;
	SDL_Surface* surf;
	short x, y;
	int top, left, bottom, right;
	int pts[4];
	Uint8 rgba[4];
	Uint32 color;
	int closed;
	int result, loop, length, drawn;

	/*get all the arguments*/
	if(!PyArg_ParseTuple(arg, "O!OOO", &PySurface_Type, &surfobj, &colorobj, &closedobj, &points))
		return NULL;
	surf = PySurface_AsSurface(surfobj);

	if(surf->format->BytesPerPixel <= 0 || surf->format->BytesPerPixel > 4)
		return RAISE(PyExc_ValueError, "unsupport bit depth for line draw");

	if(PyInt_Check(colorobj))
		color = (Uint32)PyInt_AsLong(colorobj);
	else if(RGBAFromObj(colorobj, rgba))
		color = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
	else
		return RAISE(PyExc_TypeError, "invalid color argument");

	closed = PyObject_IsTrue(closedobj);

	if(!PySequence_Check(points))
		return RAISE(PyExc_TypeError, "points argument must be a sequence of number pairs");
	length = PySequence_Length(points);
	if(length < 2)
		return RAISE(PyExc_ValueError, "points argument must contain more than 1 points");

	item = PySequence_GetItem(points, 0);
	result = TwoShortsFromObj(item, &x, &y);
	Py_DECREF(item);
	if(!result) return RAISE(PyExc_TypeError, "points must be number pairs");

	pts[0] = left = right = x;
	pts[1] = top = bottom = y;

	if(!PySurface_Lock(surfobj)) return NULL;

	drawn = 1;
	for(loop = 1; loop < length; ++loop)
	{
		item = PySequence_GetItem(points, loop);
		result = TwoShortsFromObj(item, &x, &y);
		Py_DECREF(item);
		if(!result) continue; /*note, we silently skip over bad points :[ */
		++drawn;
		pts[2] = x;
		pts[3] = y;

		if(clip_and_draw_line(surf, &surf->clip_rect, color, pts))
		{
			left = min(min(pts[0], pts[2]), left);
			top = min(min(pts[1], pts[3]), top);
			right = max(max(pts[0], pts[2]), right);
			bottom = max(max(pts[1], pts[3]), bottom);
		}

		pts[0] = pts[2];
		pts[1] = pts[3];
	}
	if(closed && drawn > 2)
	{
		item = PySequence_GetItem(points, 0);
		result = TwoShortsFromObj(item, &x, &y);
		Py_DECREF(item);
		if(result)
		{
			pts[2] = x;
			pts[3] = y;
			clip_and_draw_line(surf, &surf->clip_rect, color, pts);
		}
	}


	if(!PySurface_Unlock(surfobj)) return NULL;

	/*compute return rect*/
	return PyRect_New4((short)left, (short)top, (short)(right-left+1), (short)(bottom-top+1));
}

    /*DOC*/ static char doc_rectangle[] =
    /*DOC*/    "pygame.draw.rectangle(Surface, closed, start, end, foregoundcolor, optional backgroundcolor) -> Rect\n"
    /*DOC*/    "Draw a rectangle on a surface\n"
    /*DOC*/    "\n"
    /*DOC*/    "Draws a rectangle from points start to end on a surface.\n"
    /*DOC*/    "\n"
	/*DOC*/    "This will respect the clipping rectangle. A bounding box of the\n"
	/*DOC*/    "effected area is returned as a rectangle.\n"
    /*DOC*/    "\n"
    /*DOC*/    "Both color arguments can be either a RGB sequence or mapped color integer.\n"
    /*DOC*/    "If a background color is set, the rectangle will be filled..\n"
    /*DOC*/    "\n"
    /*DOC*/    "This function will temporarily lock the surface.\n"
    /*DOC*/ ;

static PyObject* rectangle(PyObject* self, PyObject* arg)
{
	PyObject *surfobj, *fgcolorobj, *bgcolorobj = 0, *start, *end;
	SDL_Surface *surf;
	Uint32 fgcolor, bgcolor;
	Uint8 rgba[4];
	short startx, endx, starty, endy;
	int anydraw;
	int top, left, bottom, right;

	int filled = 0;

	/*get all the arguments*/
	if(!PyArg_ParseTuple(arg, "O!OOO|O", &PySurface_Type, &surfobj, &start, &end, &fgcolorobj, &bgcolorobj))
		return NULL;
	surf = PySurface_AsSurface(surfobj);

	if(surf->format->BytesPerPixel <= 0 || surf->format->BytesPerPixel > 4)
		return RAISE(PyExc_ValueError, "unsupport bit depth for line draw");

	if(PyInt_Check(fgcolorobj))
		fgcolor = (Uint32)PyInt_AsLong(fgcolorobj);
	else if(RGBAFromObj(fgcolorobj, rgba))
		fgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
	else
		return RAISE(PyExc_TypeError, "Invalid foreground color argument.");

	if(bgcolorobj)
	{
		filled = 1;
		if(PyInt_Check(bgcolorobj))
		{
			bgcolor = (Uint32)PyInt_AsLong(bgcolorobj);
		}else{
			if(RGBAFromObj(bgcolorobj, rgba))
			{
			bgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
			}else{
				return RAISE(PyExc_TypeError, "Invalid background color argument.");
			}
		}
	}

	if(!TwoShortsFromObj(start, &startx, &starty))
		return RAISE(PyExc_TypeError, "Invalid start position argument.");
	if(!TwoShortsFromObj(end, &endx, &endy))
		return RAISE(PyExc_TypeError, "Invalid end position argument.");
	

	if(!PySurface_Lock(surfobj)) return NULL;

	if (filled)
		anydraw = drawrectanglefilled(surf, fgcolor, bgcolor, startx, starty, endx, endy);
	else
		anydraw = drawrectangle(surf, fgcolor, startx, starty, endx, endy);

	if(!PySurface_Unlock(surfobj)) return NULL;


	/*compute return rect*/
	if(!anydraw)
		return PyRect_New4(startx, starty, 0, 0);
	if(startx < endx)
	{
		left = startx;
		right = endx;
	}
	else
	{
		left = endx;
		right = startx;
	}
	if(starty < endy)
	{
		top = starty;
		bottom = endy;
	}
	else
	{
		top = endy;
		bottom = starty;
	}
	return PyRect_New4((short)left, (short)top, (short)(right-left+1), (short)(bottom-top+1));
}

    /*DOC*/ static char doc_ellipse[] =
    /*DOC*/    "pygame.draw.ellipse(Surface, closed, origin, radiusx, radiusy, foregoundcolor, optional backgroundcolor) -> Rect\n"
    /*DOC*/    "Draw an ellipse on a surface\n"
    /*DOC*/    "\n"
    /*DOC*/    "Draws an ellipse centered at origin with set radii on a surface.\n"
    /*DOC*/    "\n"
	/*DOC*/    "This will respect the clipping rectangle. A bounding box of the\n"
	/*DOC*/    "effected area is returned as a rectangle.\n"
    /*DOC*/    "\n"
    /*DOC*/    "Both color arguments can be either a RGB sequence or mapped color integer.\n"
    /*DOC*/    "If a background color is set, the ellipse will be filled..\n"
    /*DOC*/    "\n"
    /*DOC*/    "This function will temporarily lock the surface.\n"
    /*DOC*/ ;

static PyObject* ellipse(PyObject* self, PyObject* arg)
{
	PyObject *surfobj, *fgcolorobj, *bgcolorobj = 0, *origin;
	SDL_Surface *surf;
	Uint32 fgcolor, bgcolor;
	Uint8 rgba[4];
	short originx, originy, radiusx, radiusy;
	int anydraw;
	int top, left, bottom, right;

	int filled = 0;

	/*get all the arguments*/
	if(!PyArg_ParseTuple(arg, "O!OiiO|O", &PySurface_Type, &surfobj, &origin, &radiusx, &radiusy, &fgcolorobj, &bgcolorobj))
		return NULL;
	surf = PySurface_AsSurface(surfobj);

	if(surf->format->BytesPerPixel <= 0 || surf->format->BytesPerPixel > 4)
		return RAISE(PyExc_ValueError, "Unsupportorted bit depth for ellipse.");

	if(PyInt_Check(fgcolorobj))
		fgcolor = (Uint32)PyInt_AsLong(fgcolorobj);
	else if(RGBAFromObj(fgcolorobj, rgba))
		fgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
	else
		return RAISE(PyExc_TypeError, "Invalid foreground color argument.");

	if(bgcolorobj)
	{
		filled = 1;
		if(PyInt_Check(bgcolorobj))
		{
			bgcolor = (Uint32)PyInt_AsLong(bgcolorobj);
		}else{
			if(RGBAFromObj(bgcolorobj, rgba))
			{
			bgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
			}else{
				return RAISE(PyExc_TypeError, "Invalid background color argument.");
			}
		}
	}

	if(!TwoShortsFromObj(origin, &originx, &originy))
		return RAISE(PyExc_TypeError, "Invalid start position argument.");
	//if(!TwoShortsFromObj(radius, &radiusx, &radiusy))
	//	return RAISE(PyExc_TypeError, "Invalid end position argument.");
	
	printf("a");

	if(!PySurface_Lock(surfobj)) return NULL;

	if (filled)
		anydraw = drawellipsefilled(surf, fgcolor, bgcolor, originx, originy, radiusx, radiusy);
	else
		anydraw = drawellipse(surf, fgcolor, originx, originy, radiusx, radiusy);

	if(!PySurface_Unlock(surfobj)) return NULL;


	/*compute return rect*/
	if(!anydraw)
		return PyRect_New4(originx, originy, 0, 0);
	left = originx - radiusx;
	right = originx + radiusx;
	top = originy - radiusy;
	bottom = originy + radiusy;
	return PyRect_New4((short)left, (short)top, (short)(right-left+1), (short)(bottom-top+1));
}

    /*DOC*/ static char doc_triangle[] =
    /*DOC*/    "pygame.draw.triangle(Surface, closed, point1, point2, point3, foregroundcolor, optional backgroundcolor) -> Rect\n"
    /*DOC*/    "Draw a triangle on a surface\n"
    /*DOC*/    "\n"
    /*DOC*/    "Draws a triangle defined by the three points on a surface.\n"
    /*DOC*/    "\n"
	/*DOC*/    "This will respect the clipping rectangle. A bounding box of the\n"
	/*DOC*/    "effected area is returned as a rectangle.\n"
    /*DOC*/    "\n"
    /*DOC*/    "Both color arguments can be either a RGB sequence or mapped color integer.\n"
    /*DOC*/    "If a background color is set, the triangle will be filled..\n"
    /*DOC*/    "\n"
    /*DOC*/    "This function will temporarily lock the surface.\n"
    /*DOC*/ ;

static PyObject* triangle(PyObject* self, PyObject* arg)
{
	PyObject *surfobj, *fgcolorobj, *bgcolorobj = 0, *p1, *p2, *p3;
	SDL_Surface *surf;
	Uint32 fgcolor, bgcolor;
	Uint8 rgba[4];
	short x1, y1, x2, y2, x3, y3;
	int anydraw;
	int top, left, bottom, right;

	int filled = 0;

	/*get all the arguments*/
	if(!PyArg_ParseTuple(arg, "O!OOOO|O", &PySurface_Type, &surfobj, &p1, &p2, &p3,  &fgcolorobj, &bgcolorobj))
		return NULL;
	surf = PySurface_AsSurface(surfobj);

	if(surf->format->BytesPerPixel <= 0 || surf->format->BytesPerPixel > 4)
		return RAISE(PyExc_ValueError, "unsupport bit depth for ellipse");

	if(PyInt_Check(fgcolorobj))
		fgcolor = (Uint32)PyInt_AsLong(fgcolorobj);
	else if(RGBAFromObj(fgcolorobj, rgba))
		fgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
	else
		return RAISE(PyExc_TypeError, "Invalid foreground color argument.");

	if(bgcolorobj)
	{
		filled = 1;
		if(PyInt_Check(bgcolorobj))
		{
			bgcolor = (Uint32)PyInt_AsLong(bgcolorobj);
		}else{
			if(RGBAFromObj(bgcolorobj, rgba))
			{
			bgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
			}else{
				return RAISE(PyExc_TypeError, "Invalid background color argument.");
			}
		}
	}

	if(!TwoShortsFromObj(p1, &x1, &y1))
		return RAISE(PyExc_TypeError, "Invalid start position argument.");
	if(!TwoShortsFromObj(p2, &x2, &y2))
		return RAISE(PyExc_TypeError, "Invalid second position argument.");
	if(!TwoShortsFromObj(p3, &x3, &y3))
		return RAISE(PyExc_TypeError, "Invalid third position argument.");
	
	printf("a");

	if(!PySurface_Lock(surfobj)) return NULL;

	if (filled)
		anydraw = drawtrianglefilled(surf, fgcolor, bgcolor, x1, y1, x2, y2, x3, y3);
	else
		anydraw = drawtriangle(surf, fgcolor, x1, y1, x2, y2, x3, y3);

	if(!PySurface_Unlock(surfobj)) return NULL;


	/*compute return rect*/
	if(!anydraw)
		return PyRect_New4(0, 0, 0, 0);
	left = min(x1,min(x2,x3));
	right = max(x2,max(x2,x3));
	top = min(y1,min(y2,y3));
	bottom = max(y1,max(y2,y3));
	return PyRect_New4((short)left, (short)top, (short)(right-left+1), (short)(bottom-top+1));
}

    /*DOC*/ static char doc_polygon[] =
    /*DOC*/    "pygame.draw.polygon(Surface, closed, point_array, foregroundcolor, backgroundcolor) -> Rect\n"
    /*DOC*/    "Draw a polygon on a surface\n"
    /*DOC*/    "\n"
    /*DOC*/    "Draws a polygon defined by the sequence on a surface. You must pass at least three points\n"
    /*DOC*/    "in the sequence of points. Polygons are always closed, and \n"
    /*DOC*/    "a line will be draw between the first and last points.\n"
    /*DOC*/    "\n"
	/*DOC*/    "This will respect the clipping rectangle. A bounding box of the\n"
	/*DOC*/    "effected area is returned as a rectangle.\n"
    /*DOC*/    "\n"
    /*DOC*/    "Both color arguments can be either a RGB sequence or mapped color integer.\n"
    /*DOC*/    "If a background color is set, the polygon will be filled..\n"
    /*DOC*/    "\n"
    /*DOC*/    "This function will temporarily lock the surface.\n"
    /*DOC*/ ;

static PyObject* polygon(PyObject* self, PyObject* arg)
{
	PyObject *surfobj, *fgcolorobj, *bgcolorobj = 0, *points, *item;
	SDL_Surface *surf;
	Uint32 fgcolor, bgcolor;
	Uint8 rgba[4];
	short x, y;
	int  result;
	int anydraw;
	int top=0, left=0, bottom=0, right=0;
	int loop, length;
	int *xpoints;
	int *ypoints;
	int filled = 0;

	/*get all the arguments*/
	if(!PyArg_ParseTuple(arg, "O!OO|O", &PySurface_Type, &surfobj, &points,  &fgcolorobj, &bgcolorobj))
		return NULL;
	surf = PySurface_AsSurface(surfobj);

	if(surf->format->BytesPerPixel <= 0 || surf->format->BytesPerPixel > 4)
		return RAISE(PyExc_ValueError, "unsupport bit depth for ellipse");

	if(PyInt_Check(fgcolorobj))
		fgcolor = (Uint32)PyInt_AsLong(fgcolorobj);
	else if(RGBAFromObj(fgcolorobj, rgba))
		fgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
	else
		return RAISE(PyExc_TypeError, "Invalid foreground color argument.");

	if(bgcolorobj)
	{
		filled = 1;
		if(PyInt_Check(bgcolorobj))
		{
			bgcolor = (Uint32)PyInt_AsLong(bgcolorobj);
		}else{
			if(RGBAFromObj(bgcolorobj, rgba))
			{
			bgcolor = SDL_MapRGBA(surf->format, rgba[0], rgba[1], rgba[2], rgba[3]);
			}else{
				return RAISE(PyExc_TypeError, "Invalid background color argument.");
			}
		}
	}
	

	length = PySequence_Length(points);
	if(length < 4)
		return RAISE(PyExc_ValueError, "Points argument must contain more than 3 points");

	xpoints = (int*)malloc(length * sizeof(int*));
	ypoints = (int*)malloc(length * sizeof(int*));

	for(loop = 0; loop < length; ++loop)
	{
		item = PySequence_GetItem(points, loop);
		result = TwoShortsFromObj(item, &x, &y);
		Py_DECREF(item);
		if(!result) continue; /*note, we silently skip over bad points :[ */
		xpoints[loop] = (int)x;
		ypoints[loop] = (int)y;
		left = min((int)x, left);
		top = min((int)y, top);
		right = max((int)x, right);
		bottom = max((int)y, bottom);
	}	

	
	if(!PySurface_Lock(surfobj)) return NULL;
	//
	if (filled)
		anydraw = drawpolygonfilled(surf, fgcolor, bgcolor, length, xpoints, ypoints);
	else
		anydraw = drawpolygon(surf, fgcolor, length, xpoints, ypoints);

	if(!PySurface_Unlock(surfobj)) return NULL;

	free(xpoints);
	xpoints = NULL;
	free(ypoints);
	ypoints = NULL;
          	
	/*compute return rect*/
	if(!anydraw)
		return PyRect_New4(0, 0, 0, 0);
	return PyRect_New4((short)left, (short)top, (short)(right-left+1), (short)(bottom-top+1));
}


static PyMethodDef draw_builtins[] =
{
	{ "line", line, 1, doc_line },
	{ "lines", lines, 1, doc_lines },
	{ "rectangle", rectangle, 1, doc_rectangle},
	{ "ellipse", ellipse, 1, doc_ellipse},
	{ "triangle", triangle, 1, doc_triangle},
	{ "polygon", polygon, 1, doc_polygon},
	{ NULL, NULL }
};

    /*DOC*/ static char doc_pygame_draw_MODULE[] =
    /*DOC*/    "Contains routines to draw onto a surface.\n"
    /*DOC*/    "\n"
    /*DOC*/    "Note that all\n"
    /*DOC*/    "drawing routines use direct pixel access, so the surfaces\n"
    /*DOC*/    "must be locked for use. The draw functions will temporarily\n"
    /*DOC*/    "lock the surface if needed, but if performing many drawing\n"
    /*DOC*/    "routines together, it would be best to surround the drawing\n"
    /*DOC*/    "code with a lock/unlock pair.\n"
    /*DOC*/ ;

PYGAME_EXPORT
void initdraw(void)
{
	PyObject *module, *dict;

    /* create the module */
	module = Py_InitModule3("draw", draw_builtins, doc_pygame_draw_MODULE);
	dict = PyModule_GetDict(module);

	/*imported needed apis*/
	import_pygame_base();
	import_pygame_rect();
	import_pygame_surface();
}

