/*
    pygame - Python Game Library
    Copyright (C) 2000-2001  Pete Shinners

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Pete Shinners
    pete@shinners.org
*/

/* Primitive drawing tools*/

#include "pygame.h"
#include "draw.h"


/*this line clipping based heavily off of code from
http://www.ncsa.uiuc.edu/Vis/Graphics/src/clipCohSuth.c */
#define LEFT_EDGE   0x1
#define RIGHT_EDGE  0x2
#define BOTTOM_EDGE 0x4
#define TOP_EDGE    0x8
#define INSIDE(a)   (!a)
#define REJECT(a,b) (a&b)
#define ACCEPT(a,b) (!(a|b))

void drawpixel(SDL_Surface *Surface, Uint32 pixel, int x, int y)
{
	int bpp = Surface->format->BytesPerPixel;
	Uint8 *p = (Uint8 *)Surface->pixels + y * Surface->pitch + x * bpp;
	
	if (x > 0 && y > 0 && x < Surface->w-1 && y < Surface->h-1)
	{
		switch(bpp) {
		case 1:
			*p = pixel;
			break;
			
		case 2:
			*(Uint16 *)p = pixel;
			break;
			
		case 3:
			if(SDL_BYTEORDER == SDL_BIG_ENDIAN) {
				p[0] = (pixel >> 16) & 0xff;
				p[1] = (pixel >> 8) & 0xff;
				p[2] = pixel & 0xff;
			} else {
				p[0] = pixel & 0xff;
				p[1] = (pixel >> 8) & 0xff;
				p[2] = (pixel >> 16) & 0xff;
			}
			break;
			
		case 4:
			*(Uint32 *)p = pixel;
			break;
		}
	}
}

int clip_and_draw_line(SDL_Surface* surf, SDL_Rect* rect, Uint32 color, int* pts)
{
	if(!clipline(pts, rect->x, rect->y, rect->x+rect->w-1, rect->y+rect->h-1))
		return 0;
	if(pts[1] == pts[3])
		drawhorzline(surf, color, pts[0], pts[1], pts[2]);
	else
		drawline(surf, color, pts[0], pts[1], pts[2], pts[3]);
	return 1;
}

int encode(int x, int y, int left, int top, int right, int bottom)
{
	int code = 0;
	if(x < left)  code |= LEFT_EDGE;
	if(x > right) code |= RIGHT_EDGE;
	if(y < top)   code |= TOP_EDGE;
	if(y > bottom)code |= BOTTOM_EDGE;
	return code;
}

int clipline(int* pts, int left, int top, int right, int bottom)
{
	int x1 = pts[0];
	int y1 = pts[1];
	int x2 = pts[2];
	int y2 = pts[3];
	int code1, code2;
	int draw = 0;
	int swaptmp;
	float m; /*slope*/

	while(1)
	{
		code1 = encode(x1, y1, left, top, right, bottom);
		code2 = encode(x2, y2, left, top, right, bottom);
		if(ACCEPT(code1, code2)) {
			draw = 1;
			break;
		}
		else if(REJECT(code1, code2))
			break;
		else {
			if(INSIDE(code1)) {
				swaptmp = x2; x2 = x1; x1 = swaptmp;
				swaptmp = y2; y2 = y1; y1 = swaptmp;
				swaptmp = code2; code2 = code1; code1 = swaptmp;
			}
			if(x2 != x1)      
				m = (y2 - y1) / (float)(x2 - x1);
			else
				m = 1.0f;
			if(code1 & LEFT_EDGE) {
				y1 += (int)((left - x1) * m);
				x1 = left; 
			} 
			else if(code1 & RIGHT_EDGE) {
				y1 += (int)((right - x1) * m);
				x1 = right; 
			} 
			else if(code1 & BOTTOM_EDGE) {
				if(x2 != x1)
					x1 += (int)((bottom - y1) / m);
				y1 = bottom;
			} 
			else if(code1 & TOP_EDGE) {
				if(x2 != x1)
					x1 += (int)((top - y1) / m);
				y1 = top;
			} 
		}
	}
	if(draw) {
		pts[0] = x1; pts[1] = y1;
		pts[2] = x2; pts[3] = y2;
	}
	return draw;
}

/*here's my sdl'ized version of bresenham*/
void drawline(SDL_Surface* surf, Uint32 color, int x1, int y1, int x2, int y2)
{
	int deltax, deltay, signx, signy;
	int pixx, pixy;
	int x = 0, y = 0;
	int swaptmp;
	Uint8 *pixel;
	Uint8 *colorptr;

 	deltax = x2 - x1;
 	deltay = y2 - y1;
 	signx = (deltax < 0) ? -1 : 1;
 	signy = (deltay < 0) ? -1 : 1;
 	deltax = signx * deltax + 1;
 	deltay = signy * deltay + 1;

	pixx = surf->format->BytesPerPixel;
	pixy = surf->pitch;
	pixel = ((Uint8*)surf->pixels) + pixx * x1 + pixy * y1;

	pixx *= signx;
	pixy *= signy;
	if(deltax < deltay) /*swap axis if rise > run*/
	{
		swaptmp = deltax; deltax = deltay; deltay = swaptmp;
		swaptmp = pixx; pixx = pixy; pixy = swaptmp;
	}

	switch(surf->format->BytesPerPixel)
	{
	case 1:
		for(; x < deltax; x++, pixel += pixx) {
			*pixel = (Uint8)color;
			y += deltay; if(y >= deltax) {y -= deltax; pixel += pixy;}
		}break;
	case 2:
		for(; x < deltax; x++, pixel += pixx) {
			*(Uint16*)pixel = (Uint16)color;
			y += deltay; if(y >= deltax) {y -= deltax; pixel += pixy;}
		}break;
	case 3:
		if(SDL_BYTEORDER == SDL_BIG_ENDIAN) color <<= 8;
		colorptr = (Uint8*)&color;
		for(; x < deltax; x++, pixel += pixx) {
			pixel[0] = colorptr[0];
			pixel[1] = colorptr[1];
			pixel[2] = colorptr[2];
			y += deltay; if(y >= deltax) {y -= deltax; pixel += pixy;}
		}break;
	default: /*case 4*/
		for(; x < deltax; x++, pixel += pixx) {
	   *(Uint32*)pixel = (Uint32)color;
			y += deltay; if(y >= deltax) {y -= deltax; pixel += pixy;}
		}break;
	}
}

void drawhorzline(SDL_Surface* surf, Uint32 color, int x1, int x2, int y)
{
	Uint8 *pixel, *end;
	Uint8 *colorptr;

	if(x1 == x2) return;

	x1 = max(0,x1);
	x2 = max(0,x2);
	if (y < 0)
		return;

	x1 = min(surf->w,x1);
	x2 = min(surf->w,x2);
	if (y > surf->h)
		return;

	pixel = ((Uint8*)surf->pixels) + surf->pitch * y;

	if(x1 < x2)
	{
		end = pixel + x2 * surf->format->BytesPerPixel;
		pixel += x1 * surf->format->BytesPerPixel;
	}
	else
	{
		end = pixel + x1 * surf->format->BytesPerPixel;
		pixel += x2 * surf->format->BytesPerPixel;
	}

	switch(surf->format->BytesPerPixel)
	{
	case 1:
		for(; pixel <= end; ++pixel) {
			*pixel = (Uint8)color;
		}break;
	case 2:
		for(; pixel <= end; pixel+=2) {
			*(Uint16*)pixel = (Uint16)color;
		}break;
	case 3:
		if(SDL_BYTEORDER == SDL_BIG_ENDIAN) color <<= 8;
		colorptr = (Uint8*)&color;
		for(; pixel <= end; pixel+=3) {
			pixel[0] = colorptr[0];
			pixel[1] = colorptr[1];
			pixel[2] = colorptr[2];
		}break;
	default: /*case 4*/
		for(; pixel <= end; pixel+=4) {
			*(Uint32*)pixel = (Uint32)color;
		}break;
	}
}

int drawvertline(SDL_Surface *Surface, Uint32 Color, Sint32 Y1, Sint32 Y2, Sint32 X)
{
	Sint32 temp;
	
	if (Y2 < Y1){
		temp = Y2;
		Y2 = Y1;
		Y1 = temp;
	}
		
	for (temp = Y1; temp <= Y2; temp++)
		drawpixel(Surface, Color, X, temp);
		
	return 0;
}

int drawrectangle (SDL_Surface *Surface, Uint32 Color, Sint32 X1, Sint32 Y1, Sint32 X2, Sint32 Y2)
{
	Sint32 temp;
	
	if (X2 < X1){
		temp = X2;
		X2 = X1;
		X1 = temp;
	}
	if (Y2 < Y1){
		temp = Y2;
		Y2 = Y1;
		Y1 = temp;
	}
		
	drawhorzline(Surface, Color, X1, X2, Y1);
	drawvertline(Surface, Color, Y1, Y2, X1);
	drawvertline(Surface, Color, X1, X2, Y2);
	drawhorzline(Surface, Color, Y1, X2, X2);
		
	return 1;
}

int drawrectanglefilled (SDL_Surface *Surface, Uint32 Foreground, Uint32 Background, Sint32 X1, Sint32 Y1, Sint32 X2, Sint32 Y2)
{
	SDL_Rect temp;
	
	temp.x = X1;
	temp.y = Y1;
	temp.w = X2-X1;
	temp.h = Y2-Y1;
		
	SDL_FillRect(Surface, &temp, Background);
		
	if (Foreground != Background)
	{
		drawrectangle(Surface,Foreground,X1,Y1,X2,Y2);
	}
	
	return 1;
}

/* Bresenham's */
int drawellipse (SDL_Surface *Surface, Uint32 Color, Sint32 X, Sint32 Y, Sint32 XRad, Sint32 YRad)
{
	int a2, b2;
	int x, y, dec;
	
	if ((X-XRad) > (Surface->w-1))
		return 0;
	if ((Y-YRad) > (Surface->h-1))
		return 0;
	if (((X-XRad)+(XRad*2)) < 0)
		return 0;
	if (((Y-YRad)+(YRad*2)) < 0)
		return 0;
		
	a2 = XRad*XRad;
	b2 = YRad*YRad;
	
	for (x = 0, y = YRad, dec = 2*b2+a2*(1-2*YRad); b2*x <= a2*y; x++)
	{
		drawpixel(Surface, Color, X+x, Y+y);
		drawpixel(Surface, Color, X-x, Y+y);
		drawpixel(Surface, Color, X+x, Y-y);
		drawpixel(Surface, Color, X-x, Y-y);
		
		if (dec >= 0)
			dec += 4*a2*(1-(y--));
		dec += b2*(4*x+6);
	}
	
	for (x = XRad, y = 0, dec = 2*a2+b2*(1-2*XRad); a2*y <= b2*x; y++)
	{
		drawpixel(Surface, Color, X+x, Y+y);
		drawpixel(Surface, Color, X-x, Y+y);
		drawpixel(Surface, Color, X+x, Y-y);
		drawpixel(Surface, Color, X-x, Y-y);
		
		if (dec >= 0)
			dec += 4*b2*(1-(x--));
		dec += a2*(4*y+6);
	}
		
	return 1;
}

int drawellipsefilled (SDL_Surface *Surface, Uint32 Foreground, Uint32 Background, Sint32 X, Sint32 Y, Sint32 XRad, Sint32 YRad)
{
	int a2, b2;
	int x, y, dec;
	
	if ((X-XRad) > (Surface->w-1))
		return 0;
	if ((Y-YRad) > (Surface->h-1))
		return 0;
	if (((X-XRad)+(XRad*2)) < 0)
		return 0;
	if (((Y-YRad)+(YRad*2)) < 0)
		return 0;
	
	a2 = XRad*XRad;
	b2 = YRad*YRad;
		
	for (x = 0, y = YRad, dec = 2*b2+a2*(1-2*YRad); b2*x <= a2*y; x++)
	{
		drawhorzline(Surface, Background, X+x, X-x, Y+y);
		drawhorzline(Surface, Background, X+x, X-x, Y-y);
		
		if (dec >= 0)
			dec += 4*a2*(1-(y--));
		dec += b2*(4*x+6);
	}
	
	for (x = XRad, y = 0, dec = 2*a2+b2*(1-2*XRad); a2*y <= b2*x; y++)
	{
		drawhorzline(Surface, Background, X+x, X-x, Y+y);
		drawhorzline(Surface, Background, X+x, X-x, Y-y);
		
		if (dec >= 0)
			dec += 4*b2*(1-(x--));
		dec += a2*(4*y+6);
	}
		
	if (Foreground != Background)
	{
		drawellipse(Surface, Foreground,X,Y, XRad, YRad);
	}
	
	return 1;
}

int drawpolygon (SDL_Surface *Surface, Uint32  Color, Sint32 Vertices, Sint32 *X, Sint32 *Y)
{
	Sint32 v;
	
	/* Doesn't work for concave polygons! */
	for (v=0; (v+1) < Vertices; v++) {
		drawline(Surface, Color, X[v], Y[v], X[v+1], Y[v+1]);
	}
	drawline(Surface, Color, X[0], Y[0], X[Vertices-1], Y[Vertices-1]);
	
	return 1;
}

int drawpolygonfilled (SDL_Surface *Surface, Uint32 Foreground, Uint32 Background, Sint32 Vertices, Sint32 *X, Sint32 *Y)
{
	Sint32 v;
	
	/* Doesn't work for concave polygons! */
	for (v=0; (v+2) < Vertices; v++) {
		drawtrianglefilled (Surface, Background, Background, X[0], Y[0], X[v+1], Y[v+1], X[v+2], Y[v+2]);
	}
	
	if (Foreground != Background)
	{
		drawpolygon(Surface, Foreground,Vertices, X, Y);
	}
	
	return 1;
}

int drawtriangle (SDL_Surface *Surface, Uint32 Color, Sint32 X1, Sint32 Y1, Sint32 X2, Sint32 Y2, Sint32 X3, Sint32 Y3)
{
	drawline(Surface, Color, X1, Y1, X2, Y2);
	drawline(Surface, Color, X2, Y2, X3, Y3);
	drawline(Surface, Color, X3, Y3, X1, Y1);
	
	return 1;
}


int drawtrianglefilled (SDL_Surface *Surface, Uint32 Foreground, Uint32 Background, Sint32 x1, Sint32 y1, Sint32 x2, Sint32 y2, Sint32 x3, Sint32 y3)
{
	Sint16 xl,xs,y,t;
	
	if ( y1 > y2 ) {
		SWAP(y1,y2,y);
		SWAP(x1,x2,y);
	}
	
	if ( y2 > y3 ) {
		SWAP(y2,y3,y);
		SWAP(x2,x3,y);
	}
	
	if ( y1 > y2 ) {
		SWAP(y1,y2,y);
		SWAP(x1,x2,y);
	}
	
	for ( y = y1; y <= y3; y++) {
		if ( y > y2 ) {
			t = (y2-y3);
			t = t?t:1;
			xs = ((x2-x3)*(y-y2))/t+x2;
		}
		else {
			t = (y1-y2);
			t = t?t:1;
			xs = ((x1-x2)*(y-y1))/t+x1;
		}
		t = (y1-y3);
		xl = ((x1-x3)*(y-y1))/((t==0)?1:t)+x1;
		
		drawhorzline(Surface,Background ,xl,xs,y);
	}

	if (Foreground != Background)
	{
		drawtriangle(Surface,Foreground,x1,y1,x2,y2,x3,y3);
	}

	return 1;
}
/*int drawtrianglefilled (SDL_Surface *Surface, Uint32 Foreground, Uint32 Background, Sint32 X1, Sint32 Y1, Sint32 X2, Sint32 Y2, Sint32 X3, Sint32 Y3)
{
	Sint32 y;
	
	int topx, topy;
	int midx, midy;
	int botx, boty;
	int ebleft, ebright;
	int rpos, lpos;
	
	midx = 0;
	midy = 0;
	
	// Sort points by Y-order 
	if (Y2 <= Y1) {
		topx = X2;
		topy = Y2;
		botx = X1;
		boty = Y1;
	}
	else if (Y2 > Y1) {
		topx = X1;
		topy = Y1;
		botx = X2;
		boty = Y2;
	}
	
	if (Y3 >= topy && Y3 <= boty) {
		midx = X3;
		midy = Y3;
	}
	else if (Y3 < topy) {
		midx = topx;
		midy = topy;
		topx = X3;
		topy = Y3;
	}
	else if (Y3 > boty) {
		midx = botx;
		midy = boty;
		botx = X3;
		boty = Y3;
	}
	
	ebleft = (midx - topx) / (midy - topy);
	ebright = (botx - topx) / (boty - topy);
	
	rpos = topx;
	lpos = topx;
	
	for (y = topy; y <= midy; y++) {
		drawhorzline(Surface, Background, rpos, lpos, y);
		rpos += ebright;
		lpos += ebleft;
	}
	
	ebleft = (botx - midx) / (boty - midy);
	
	lpos = midx;
	
	for (y = midy; y <= boty; y++) {
		drawhorzline(Surface, Background, rpos, lpos, y);
		rpos += ebright;
		lpos += ebleft;
	}
	
	if (Foreground != Background)
	{
		drawtriangle(Surface,Foreground,X1,Y1,X2,Y2,X3,Y3);
	}
	
	return 1;
}*/


#if 0
/*CODE I PLAN TO IMPLEMENT IN THE FUTURE*/
/*along with this ellipse stuff, a triangle would be keen,
from triangle a simple polygons will be easy too, but needed?*/

/*anyone feel free to contribute :] */
Ok... The circle algorithm:

        x = 0
        y = radius
        switch = 3 - 2*radius

    loop:

        plot (x,y): plot (x,-y): plot (-x,y): plot (-x,-y)
        plot (y,x): plot (y,-x): plot (-y,x): plot (-y,-x)

        if switch < 0 then
            switch = switch + 4 * x + 6
        else
            switch = switch + 4 * (x - y) + 10
            y = y - 1

        x = x + 1

        if x <= y then goto loop

        end



void symmetry(x,y)
int x,y;
{
 PUT_PIXEL(+x,+y);  // This would obviously be inlined!
 PUT_PIXEL(-x,+y);  // and offset by a constant amount
 PUT_PIXEL(-x,-y);
 PUT_PIXEL(+x,-y);
}

void bresenham_ellipse(a,b) /*a = xradius, b= yradius*/
int a,b;
{
 int x,y,a2,b2, S, T;

 a2 = a*a;
 b2 = b*b;
 x = 0;
 y = b;
 S = a2*(1-2*b) + 2*b2;
 T = b2 - 2*a2*(2*b-1);
 symmetry(x,y);
 do
   {
    if (S<0)
       {
        S += 2*b2*(2*x+3);
        T += 4*b2*(x+1);
        x++;
       }
      else if (T<0)
          {
           S += 2*b2*(2*x+3) - 4*a2*(y-1);
           T += 4*b2*(x+1) - 2*a2*(2*y-3);
           x++;
           y--;
          }
         else
          {
           S -= 4*a2*(y-1);
           T -= 2*a2*(2*y-3);
           y--;
          }
    symmetry(x,y);
   }
 while (y>0);
}
#endif
