import os, sys, random
import iniparse
import config
from data import *
from dice import DiceBag

class Map:

	def __init__(self, map, screen):

		# Needed data
		self.screen = screen
		self.layers = []
		self.tile_x = 32 # Width of map tiles in pixels
		self.tile_y = 32 # Height of map tiles in pixels
		self.tilesheet = load_sprite('tilemap.png')
		self.tiles = self.tilesheet.images([ \
			(224, 64, self.tile_x, self.tile_y),       # 0 Grass
			(32, 0, self.tile_x, self.tile_y),      # 1 Grass detail
			(64, 0, self.tile_x, self.tile_y),      # 2 Grass detail
			(96, 0, self.tile_x, self.tile_y),      # 3 Grass detail
			(128, 0, self.tile_x, self.tile_y),     # 4 Grass detail
			(32, 32, self.tile_x, self.tile_y),     # 5 Grass detail
			(64, 32, self.tile_x, self.tile_y),     # 6 Grass detail
			(96, 32, self.tile_x, self.tile_y),     # 7 Grass detail
			(128, 32, self.tile_x, self.tile_y),    # 8 Mushroom
			(160, 0, self.tile_x, self.tile_y),     # 9 Bush 1
			(192, 0, self.tile_x, self.tile_y),     # 10 Bush 2
			(224, 0, self.tile_x, self.tile_y),     # 11 Bush 3
			(0, 32, self.tile_x, self.tile_y),      # 12 Bush 4
			(0, 64, self.tile_x, self.tile_y*2),    # 13 Bush 5
			(32, 64, self.tile_x, self.tile_y*2),   # 14 Bush 6
			(160, 32, self.tile_x, self.tile_y),    # 15 Rock
			(192, 32, self.tile_x, self.tile_y),    # 16 Rock
			(64, 64, self.tile_x, self.tile_y*2),   # 17 Tree
			(96, 64, self.tile_x*2, self.tile_y*2), # 18 Tree
			(160, 64, self.tile_x*2, self.tile_y*2) # 19 Tree
			],(255,0,255))
		self.nowalk = []
		self.danger = []
		self.mapcfg = load_mapcfg(map)
		self.base_tile = int(self.mapcfg.general.base_tile)
		self.set_above = False
		self.start_tile = [ int(self.mapcfg.general.start_tile_x), int(self.mapcfg.general.start_tile_y) ]
		self.dice = DiceBag()

		# Read map file
		self.read_map(map)

		# Scroll map to show player start location
		scroll_x, scroll_y = self.scroll_start()
		self.scrollpos = [ scroll_x, scroll_y ]

		# Shift the unwalkable rectangles according to the map's scroll position
		for rect in self.nowalk:
			rect.move(self.scrollpos[0], self.scrollpos[1])

	def read_map(self, map):
	# This reads the given map file and converts it into a list of layers

		# Read the map file's contents and config
		map_file = load_level(map)

		# Add the layers in the map file to a list
		temp_layer = []
		for line in map_file:
			if line == "\n":
				self.layers.append(temp_layer)
				temp_layer = []
			else:
				line = line.rstrip()
				temp_layer.append(line)
		map_file.close()

		# Draw each layer of tiles
		self.background = TileLayer('bg', self.tile_x, len(self.layers[0][0]), self.tile_y, len(self.layers[0]))
		self.foreground = TileLayer('fg', self.tile_x, len(self.layers[0][0]), self.tile_y, len(self.layers[0]))
		for layer in range(0, len(self.layers)-1 ):
			self.draw(layer)

	def draw(self, layernum):

		linenum = 0
		charnum = 0

		# For each layer, look up and draw tiles
		for line in self.layers[layernum]:
			for char in line:

				# Set the default background tile from the map's config
				tile = self.base_tile

				# Layer 0: Base terrain
				# This layer draws the base terrain and nothing else
				# G = grass
				# X = blocked tile
				if layernum == 0:
					if char == "G":
						tile = 0
					if char == "X":
						self.set_nowalk(charnum, linenum, self.tile_x)

				# Layer 1: Terrain detail
				# This layer draws random details on the terrain
				# G = grass detail
				if layernum == 1:
					if char == "G":
						# Grass
						if self.dice.d20.roll() >= 18:
							tile = self.create_tile(1, 7, charnum, linenum)
						# Mushroom
						if self.dice.d100.roll(2) >= 198:
							tile = 8

				# Layer 2: Terrain background
				# This layer draws terrain objects to the background layer
				# B = bush
				# T = tree
				# R = rock
				if layernum == 2:
					if char == "B":
						tile = self.create_tile(9, 14, charnum, linenum, nowalk=True, above=True)
					if char == "T":
						tile = self.create_tile(17, 19, charnum, linenum, nowalk=True, above=True)
					if char == "R":
						tile = self.create_tile(15, 16, charnum, linenum, nowalk=True)

				# Re-align bigger tiles
				self.tile_offset = [ 0, 0 ]
				if not self.tile_x == self.tiles[tile].get_width():
					self.tile_offset[0] = -(self.tiles[tile].get_width()-self.tile_x)
				if not self.tile_y == self.tiles[tile].get_height():
					self.tile_offset[1] = -(self.tiles[tile].get_height()-self.tile_y)

				# Draw each tile
				if not char == ".":
					self.draw_tile(self.tiles[tile], self.tiles[tile].get_width(), self.tiles[tile].get_height(), self.tile_offset, charnum, linenum)

				charnum+=1
			linenum+=1
			charnum = 0
		linenum = 0
		charnum = 0

	def draw_tile(self, tile, w, h, offset, col, row):
	# Draw the tile to the background or foreground

		self.background.get().blit(tile, ( (self.tile_x*col)+offset[0], (self.tile_y*row)+offset[1]) )
		if self.set_above:
			self.foreground.get().blit(tile, ( (self.tile_x*col)+offset[0], (self.tile_y*row)+offset[1]), (0,0,w,h-self.tile_y) )

	def create_tile(self, start, end, x, y, nowalk=False, above=False):
	# Generates a random tile

		tile_range = range(start,end+1)
		tile = random.choice(tile_range)
		self.set_above = above
		if nowalk:
			self.set_nowalk(x, y, self.tiles[tile].get_width())
		return tile

	def set_nowalk(self, x, y, w):
	# Sets given tile as unwalkable

		nowalk = pygame.Rect( (x*self.tile_x)-(w-self.tile_x), y*self.tile_y, w, self.tile_y )
		self.nowalk.append(nowalk)

	def scroll_start(self):
	# Scroll the map to show the player

		half_screen = [ (config.screen_size[0]/self.tile_x)/2, (config.screen_size[1]/self.tile_y)/2 ]
		scroll_x, scroll_y = 0, 0
		if self.start_tile[0] > half_screen[0]:
			scroll_x = -((self.start_tile[0] * self.tile_x) - (half_screen[0] * self.tile_x))
			while -scroll_x + config.screen_size[0] > self.background.get().get_width():
				scroll_x+=self.tile_x
		if self.start_tile[1] > half_screen[1]:
			scroll_y = -((self.start_tile[1] * self.tile_y) - (half_screen[1] * self.tile_y))
			while -scroll_y + config.screen_size[1] > self.background.get().get_height():
				scroll_y+=self.tile_y
		return scroll_x, scroll_y


class TileLayer(pygame.sprite.DirtySprite):
	def __init__(self, layer, w, numx, h, numy):
		pygame.sprite.DirtySprite.__init__(self)
		if layer == 'fg':
			self.image = pygame.Surface((w*numx,h*numy), SRCALPHA, 32).convert_alpha()
		else:
			self.image = pygame.Surface((w*numx,h*numy)).convert()
		self.rect = self.image.get_rect()

	def get(self):
		return self.image
