/* An NS switch module for libc to route dns over Tor.
 * Install to /lib/libnss_tor.so.2 and configure /etc/nsswitch.conf with
 *    hosts: files tor [NOTFOUND=return TRYAGAIN=return] dns 
 * 
 * The module is controllde by the NSS_TOR environment variable:
 *    NSS_TOR=always  -> never use dns, if Tor is unavaible cause dns failure
 *    NSS_TOR=usually -> use Tor if possible, otherwise continue
 *    everything else -> do not use Tor
 */

#include <unistd.h>
#include <nss.h>
#include <netdb.h>
#include <errno.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <string.h>

enum tor_config { ALWAYS, USUALLY, NEVER };

#define FAIL_NOTFOUND fail_notfound(config, errnop, h_errnop)
static enum nss_status fail_notfound(enum tor_config config, int *errnop, int *h_errnop) {
        *errnop = ENOENT;
        *h_errnop = HOST_NOT_FOUND;
        return NSS_STATUS_NOTFOUND;
}

#define FAIL_TRYAGAIN fail_tryagain(config, errnop, h_errnop)
static enum nss_status fail_tryagain(enum tor_config config, int *errnop, int *h_errnop) {
        *errnop = EAGAIN;
        *h_errnop = TRY_AGAIN;
        return NSS_STATUS_TRYAGAIN;
}

#define FAIL_UNAVAILABLE fail_unavailable(config, errnop, h_errnop)
static enum nss_status fail_unavailable(enum tor_config config, int *errnop, int *h_errnop) {
        if (config == ALWAYS) return FAIL_NOTFOUND;
        *errnop = ENOENT;
        *h_errnop = NO_RECOVERY;
        return NSS_STATUS_UNAVAIL;
}

enum nss_status _nss_tor_gethostbyname2_r (
    const char *name, int af,
    struct hostent *result_buf,
    char *buf, size_t buflen,
    int *errnop, int *h_errnop) {
        int sock_fd, len, alen;
        char message[1024];
        struct sockaddr_in sin;
        char* config_str;
        enum tor_config config = NEVER;
        
        /* Should the module be enabled? */
        if ((config_str = getenv("NSS_TOR")) == 0) return FAIL_UNAVAILABLE;
        if (!strcmp(config_str, "always"))  config = ALWAYS;
        if (!strcmp(config_str, "usually")) config = USUALLY;
        if (config == NEVER) return FAIL_UNAVAILABLE;
        
        if ((sock_fd = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1)
                return FAIL_TRYAGAIN;
                
        memset(&sin, 0, sizeof(sin));
        sin.sin_family = PF_INET;
        sin.sin_port = htons(9050);
        sin.sin_addr.s_addr = inet_addr("127.0.0.1");
        
        if (connect(sock_fd, (const struct sockaddr*)&sin, sizeof(sin)) < 0)
                return FAIL_UNAVAILABLE;
        
        /* v5. one method only. no auth. */
        if (write(sock_fd, "\5\1\0", 3) != 3)
                return FAIL_TRYAGAIN;
        if (read(sock_fd, message, 2) != 2)
                return FAIL_TRYAGAIN;
        
        /* expect: v5 ok */
        if (message[0] != 5 || message[1] != 0)
                return FAIL_UNAVAILABLE;
        
        len = strlen(name);
        if (len > 255) len = 255;
        
        message[0] = 5;      /* v5*/
        message[1] = '\360'; /* RESOLVE */
        message[2] = 0;      /* reserved */
        message[3] = 3;      /* type 3 = domain name */
        /* length+body */
        message[4] = strlen(name);
        memcpy(&message[5], name, len);
        /* empty port */
        message[5+len] = 0;
        message[6+len] = 0;
        if (write(sock_fd, message, len+7) != len+7)
                return FAIL_TRYAGAIN;
        
        if (read(sock_fd, message, 4) != 4)
                return FAIL_TRYAGAIN;
        
        /* expect: v5. ok. 0. atype. */
        if (message[0] != 5 || message[2] != 0)
                return FAIL_UNAVAILABLE;
        
        switch (message[1]) {
        case 0:		break;			/* OK */
        case 1:		return FAIL_TRYAGAIN;	/* general socks failure */
        case 2:		return FAIL_NOTFOUND;	/* forbidden */
        case 3:		return FAIL_NOTFOUND;	/* network unreachable */
        case 4:		return FAIL_NOTFOUND;	/* host unreachable */
        case 5:		return FAIL_NOTFOUND;	/* connect refused */
        case 6:		return FAIL_TRYAGAIN;	/* TTL expired */
        case 7:		return FAIL_UNAVAILABLE;/* request unsupported */
        case 8:		return FAIL_UNAVAILABLE;/* address type unsupported */
        default:	return FAIL_NOTFOUND;	/* ??? */
        }
        
        /* buffer: <name> <null ptr> <self ptr> <null ptr> <addr> */
        /*         len    4          4          4          alen */
        
        switch (message[3]) {
        case 1: /* IPv4 */
                result_buf->h_addrtype = PF_INET;
                alen = 4;
                break;
        case 3: /* DNS */
                return FAIL_NOTFOUND; /* wtf? why a hostname answer. */
        case 4: /* IPv6 */
                result_buf->h_addrtype = PF_INET6;
                alen = 16;
                break;
        default:
                return FAIL_NOTFOUND;	/* ??? */
        }
        
        /* Match what they want. */
        if (result_buf->h_addrtype != af)
                return FAIL_NOTFOUND;
        
        if (buflen < len + alen + 4*sizeof(char*)) {
                *errnop = ERANGE;
                *h_errnop = TRY_AGAIN;
                return NSS_STATUS_TRYAGAIN;
        }
        
        result_buf->h_name = buf;
        memcpy(buf, name, len+1);
        buf += (len+sizeof(char*)) & ~(sizeof(char*)-1);

        result_buf->h_aliases = (char**)buf;
        *((char**)buf) = 0;
        buf += sizeof(char*);
        
        result_buf->h_length = 1;
        result_buf->h_addr_list = (char**)buf;
        *((char**)buf) = buf + 2*sizeof(char*);
        buf += sizeof(char*);
        *((char**)buf) = 0;
        buf += sizeof(char*);
        
        if (read(sock_fd, buf, alen) != alen)
                return FAIL_TRYAGAIN;
        
        *errnop = 0;
        *h_errnop = 0;
        return NSS_STATUS_SUCCESS;
}

enum nss_status _nss_tor_gethostbyname_r (
    const char *name, 
    struct hostent *result_buf,
    char *buf, size_t buflen,
    int *errnop, int *h_errnop) {
          return _nss_tor_gethostbyname2_r(name, PF_INET, result_buf, buf, buflen, errnop, h_errnop);
}
